import logging
from collections import namedtuple

import requests

from sepelib.core import config
from walle import locks
from walle.clients import utils
from walle.errors import RecoverableError

Response = namedtuple("Response", ["tor", "interface", "state"])


class Status:
    OK = "ok"


log = logging.getLogger(__name__)


class VlanTogglerError(RecoverableError):
    pass


class VlanTogglerPersistentError(VlanTogglerError):
    pass


def get_port_state(switch: str, port: str) -> Response:
    return _raw_request(method="GET", params={"tor": switch, "interface": port})


def switch_port_state(switch: str, port: str, target: str) -> Response:
    with locks.VlanTogglerSwitchInterruptableLock(switch):
        return _raw_request(method="POST", params={"tor": switch, "interface": port, "state": target})


# NOTE(rocco66): see https://st.yandex-team.ru/CLOUD-81673 for details
VLAN_TOGGLER_HTTP_NON_RETRY_STATUSES = {422, 404, 403, 526, 502, 400, 511}
VLAN_TOGGLER_HTTP_RETRY_STATUSES = {523, 520, 522, 523, 507, 509}


def _should_retry(response: requests.Response) -> bool:
    if response.status_code in VLAN_TOGGLER_HTTP_RETRY_STATUSES:
        return True
    elif response.status_code in VLAN_TOGGLER_HTTP_NON_RETRY_STATUSES:
        return False
    else:
        log.info("VLAN Toggler unknown response code %s", response.status_code)
        return False


def _raw_request(method: str, params: dict[str]) -> Response:
    url = config.get_value("vlan_toggler.api_url")
    headers = {"Authorization": "Bearer " + config.get_value("vlan_toggler.access_token", default="")}

    try:
        resp = utils.json_request("vlan-toggler", method, url, headers=headers, data=params)
    except utils.HTTPError as e:
        resp = e.response
        error = "VLAN Toggler HTTP code {}.".format(resp.status_code)
        if _should_retry(resp):
            raise VlanTogglerError(error)
        else:
            raise VlanTogglerPersistentError(error)

    if resp["status"] != Status.OK:
        raise VlanTogglerError("Invalid status: {}".format(resp["status"]))

    res = resp["result"]
    return Response(tor=res["tor"], interface=res["interface"], state=res["state"])
