"""ya.subr client."""

import logging
import re
from collections import namedtuple
from ipaddress import IPv4Address, IPv4Network, IPv6Network

from cachetools.func import lru_cache

from sepelib.core.exceptions import Error, LogicalError
from walle import constants as walle_constants
from walle.errors import InvalidHostConfiguration, YaSubrRecordNotFoundError

log = logging.getLogger(__name__)

_RA_NETWORK = "RA"
_NetworkConfig = namedtuple("NetworkConfig", ["network", "bb6_net", "bb_vlan", "fb4_vlan", "fb6_net", "fb6_vlan"])


NETWORKS = """kr      5.255.238.64/27     2a02:6b8:0:411::/64     696     XXXX                XXXX    XXXX                    XXXX
ugrb    XXXX                2a02:6b8:0:253d::/64    1388    XXXX                XXXX    2a02:6b8:0:aa0::/64     761     ISS
ugrb    XXXX                2a02:6b8:b010:2048::/64 1387    XXXX                XXXX    XXXX                    XXXX    Nirvana
ugrb    5.255.220.0/23      2a02:6b8:0:2503::/64    603     XXXX                XXXX    2a02:6b8:0:a1d::/64     763
ugrb    5.45.194.0/24       2a02:6b8:0:2503::/64    603     XXXX                XXXX    2a02:6b8:0:a1d::/64     763
ugrb    5.45.244.64/26      2a02:6b8:0:253c::/64    599     XXXX                XXXX    2a02:6b8:0:a1d::/64     763     balancers
ugrb    5.45.245.32/27      2a02:6b8:0:251c::/64    1312    XXXX                XXXX    2a02:6b8:0:a04::/64     762
ugrb    37.9.80.0/22        2a02:6b8:0:2502::/64    604     XXXX                XXXX    RA                      761
ugrb    37.9.84.0/22        2a02:6b8:0:2502::/64    604     XXXX                XXXX    RA                      761
ugrb    37.9.88.176/28      2a02:6b8:b010:2001::/64 597     XXXX                XXXX    XXXX                    XXXX    stain2
ugrb    37.9.92.0/22        2a02:6b8:0:2502::/64    604     XXXX                XXXX    RA                      761
ugrb    37.140.128.0/23     2a02:6b8:0:2502::/64    604     XXXX                XXXX    RA                      761
ugrb    37.140.131.240/28   2a02:6b8:0:2535::/64    1301    XXXX                XXXX    XXXX                    XXXX
ugrb    XXXX                2a02:6b8:0:2532::/64    510     XXXX                XXXX    2a02:6b8:0:a1d::/64     763     build
ugrb    178.154.189.0/24    2a02:6b8:0:2505::/64    595     XXXX                XXXX    2a02:6b8:f000:8000::/64 763     Spider
ugrb    37.140.141.80/28    2a02:6b8:0:2522::/64    614     XXXX                XXXX    2a02:6b8:0:a1d::/64     763     ya.lite
ugrb    95.108.201.144/28   2a02:6b8:b000:4000::/64 1354    XXXX                XXXX    2a02:6b8:0:aa6::/64     742     KiKiMR
ugrb    95.108.216.0/23     2a02:6b8:0:2502::/64    604     XXXX                XXXX    RA                      761
ugrb    130.193.57.64/27    2a02:6b8:b000:4001::/64 522     XXXX                XXXX    2a02:6b8:0:a04::/64     762     CMSEARCHNETS
ivaa    XXXX                2a02:6b8:b010:5058::/64 1387    XXXX                XXXX    XXXX                    XXXX    QLOUDNETS Nirvana
ivaa    5.45.210.0/24       2a02:6b8:b010:503c::/64 690     XXXX                XXXX    XXXX                    XXXX    hitrobot
ivaa    5.45.248.64/26      2a02:6b8:b010:503e::/64 593     XXXX                XXXX    XXXX                    XXXX    antivirus
ivaa    37.9.76.0/22        2a02:6b8:0:88f::/64     594     XXXX                XXXX    2a02:6b8:0:a92::/64     866
ivaa    84.201.148.176/29   2a02:6b8:0:831::/64     555     XXXX                XXXX    XXXX                    XXXX    oops
ivaa    84.201.148.192/26   2a02:6b8:0:894::/64     599     XXXX                XXXX    2a02:6b8:0:a0d::/64     764     balancer
ivaa    93.158.142.0/23     2a02:6b8:0:888::/64     604     XXXX                XXXX    RA                      761
ivaa    95.108.128.0/24     2a02:6b8:0:88c::/64     603     XXXX                XXXX    2a02:6b8:0:a0b::/64     763
ivaa    130.193.37.0/24     2a02:6b8:0:88e::/64     595     XXXX                XXXX    2a02:6b8:f000:6000::/64 763     Spider
ivaa    95.108.179.0/24     2a02:6b8:0:88e::/64     595     XXXX                XXXX    2a02:6b8:f000:6000::/64 763     Spider
ivaa    95.108.137.32/27    2a02:6b8:b010:5040::/64 586     XXXX                XXXX    XXXX                    XXXX    antivirus dev
ivaa    95.108.137.128/27   2a02:6b8:0:887::/64     698     XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivaa    95.108.137.176/28   2a02:6b8:b000:3000::/64 1354    XXXX                XXXX    2a02:6b8:0:aa5::/64     742     KiKiMR
ivaa    95.108.142.0/24     2a02:6b8:0:80b::/64     696     XXXX                XXXX    2a02:6b8:0:a06::/64     762
ivaa    95.108.143.0/24     2a02:6b8:0:161b::/64    522     XXXX                XXXX    2a02:6b8:0:a12::/64     762     CMSEARCHNETS
ivaa    95.108.145.0/24     2a02:6b8:0:821::/64     639     XXXX                XXXX    XXXX                    XXXX    not search
ivaa    95.108.147.0/24     2a02:6b8:0:82a::/64     621     XXXX                XXXX    2a02:6b8:0:a86::/64     721     PPB
ivaa    95.108.150.0/23     2a02:6b8:0:88c::/64     603     XXXX                XXXX    2a02:6b8:0:a0b::/64     763
ivaa    93.158.152.0/24     2a02:6b8:0:88e::/64     595     XXXX                XXXX    2a02:6b8:f000:6000::/64 763     Spider
ivaa    95.108.194.128/26   2a02:6b8:b040:0::/64    1305    XXXX                XXXX    2a02:6b8:f030:2000::/64 865     RTMR
ivaa    95.108.194.192/28   2a02:6b8:0:81b::/64     1301    XXXX                XXXX    XXXX                    XXXX
ivaa    95.108.194.224/28   2a02:6b8:0:82e::/64     614     XXXX                XXXX    2a02:6b8:0:a0d::/64     764     ya.lite
ivaa    95.108.206.0/24     2a02:6b8:0:88c::/64     603     XXXX                XXXX    2a02:6b8:0:a06::/64     762
ivaa    95.108.220.128/28   2a02:6b8:0:80f::/64     643     XXXX                XXXX    2a02:6b8:0:ab0::/64     743     SUGGEST
ivaa    95.108.222.0/23     2a02:6b8:0:821::/64     639     XXXX                XXXX    XXXX                    XXXX    not search
ivaa    141.8.177.0/24      2a02:6b8:0:160f::/64    603     XXXX                XXXX    2a02:6b8:0:a12::/64     762     robot
ivaa    141.8.178.128/25    2a02:6b8:0:887::/64     698     XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivaa    213.180.198.0/24    2a02:6b8:0:888::/64     604     XXXX                XXXX    RA                      761
ivab    XXXX                2a02:6b8:0:163b::/64    1388    XXXX                XXXX    2a02:6b8:0:a12::/64     762     ISS
ivab    5.255.224.128/27    2a02:6b8:0:1622::/64    1321    XXXX                XXXX    XXXX                    XXXX
ivab    5.255.241.208/28    2a02:6b8:b010:5025::/64 1438    XXXX                XXXX    XXXX                    XXXX
ivab    87.250.252.128/27   2a02:6b8:0:161d::/64    1312    XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivab    95.108.133.32/27    2a02:6b8:0:160d::/64    509     XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivab    95.108.218.0/23     2a02:6b8:0:1611::/64    565     XXXX                XXXX    2a02:6b8:0:a91::/64     866
ivab    130.193.62.0/26     2a02:6b8:0:160e::/64    531     XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivab    130.193.63.0/24     2a02:6b8:0:160f::/64    603     XXXX                XXXX    2a02:6b8:0:a12::/64     762
ivab    141.8.173.192/27    2a02:6b8:0:894::/64     599     XXXX                XXXX    2a02:6b8:0:a12::/64     762     balancer
ivab    141.8.179.0/24      2a02:6b8:0:160b::/64    604     XXXX                XXXX    RA                      761     only porter135
ivab    178.154.132.0/22    2a02:6b8:0:160b::/64    604     XXXX                XXXX    RA                      761
ivab    178.154.184.0/23    2a02:6b8:0:160b::/64    604     XXXX                XXXX    RA                      761
myta    XXXX                2a02:6b8:b010:9005::/64 555     XXXX                XXXX    XXXX                    XXXX    OOPSNETS
mytb    XXXX                2a02:6b8:b000:9000::/64 1387    XXXX                XXXX    XXXX                    XXXX    QLOUDNETS Nirvana
myta    178.154.149.0/24    2a02:6b8:b010:9025::/64 595     XXXX                XXXX    2a02:6b8:f000:7000::/64 763     Spider
myt1    5.255.218.192/26    2a02:6b8:0:140f::/64    522     XXXX                XXXX    2a02:6b8:0:a18::/64     762     CMSEARCHNETS
myt1    84.201.134.0/23     2a02:6b8:0:149f::/64    603     XXXX                XXXX    2a02:6b8:0:a01::/64     761
myt2    77.88.1.112/29      2a02:6b8:0:1482::/64    597     XXXX                XXXX    XXXX                    XXXX    stain
myt2    84.201.158.48/28    2a02:6b8:0:1434::/64    698     XXXX                XXXX    2a02:6b8:0:a18::/64     762
myt2    87.250.237.0/26     2a02:6b8:0:140f::/64    522     XXXX                XXXX    2a02:6b8:0:a18::/64     762     CMSEARCHNETS
myt2    95.108.225.48/28    2a02:6b8:0:141d::/64    643     XXXX                XXXX    2a02:6b8:0:ab5::/64     743     SUGGEST
myt2    95.108.225.64/29    2a02:6b8:0:143e::/64    555     XXXX                XXXX    XXXX                    XXXX    oops
myt2    95.108.233.0/24     2a02:6b8:0:142a::/64    621     XXXX                XXXX    2a02:6b8:0:a80::/64     721     PPB
myt5    XXXX                2a02:6b8:0:1460::/64    1388    XXXX                XXXX    2a02:6b8:0:a11::/64     766     ISS
myt5    84.201.158.32/28    2a02:6b8:0:143b::/64    614     XXXX                XXXX    2a02:6b8:0:a0c::/64     765     ya.lite
myt5    87.250.243.64/26    2a02:6b8:0:149b::/64    599     XXXX                XXXX    2a02:6b8:0:a0c::/64     765     balancer
myt5    178.154.166.0/27    2a02:6b8:0:14a6::/64    593     XXXX                XXXX    XXXX                    XXXX
myt5    178.154.166.128/25  2a02:6b8:0:1411::/64    690     XXXX                XXXX    XXXX                    XXXX
myt5    178.154.193.0/24    2a02:6b8:0:1497::/64    604     XXXX                XXXX    RA                      761
myt5    178.154.208.0/23    2a02:6b8:0:149c::/64    603     XXXX                XXXX    2a02:6b8:0:a0c::/64     765
myt5    178.154.225.68/30   2a02:6b8:b010:7038::/64 509     XXXX                XXXX    2a02:6b8:0:a18::/64     762
myt6    178.154.140.0/23    2a02:6b8:0:1498::/64    604     XXXX                XXXX    RA                      761
myt6    178.154.150.0/24    2a02:6b8:0:142f::/64    603     XXXX                XXXX    2a02:6b8:0:a11::/64     766
myt6    178.154.151.0/24    2a02:6b8:0:142f::/64    603     XXXX                XXXX    2a02:6b8:0:a11::/64     766
myt6    213.180.212.32/28   2a02:6b8:0:1463::/64    1301    XXXX                XXXX    XXXX                    XXXX
fola    XXXX                2a02:6b8:0:c60::/64     1388    XXXX                XXXX    2a02:6b8:0:a10::/64     761     ISS
fola    XXXX                2a02:6b8:b000:1002::/64 1387    XXXX                XXXX    XXXX                    XXXX    QLOUDNETS Nirvana
fola    5.45.202.0/24       2a02:6b8:0:c25::/64     603     XXXX                XXXX    2a02:6b8:f000:5005::/64     763
fola    5.45.203.0/26       2a02:6b8:b000:1000::/64 510     XXXX                XXXX    2a02:6b8:0:a10::/64     761     build
fola    5.45.236.0/22       2a02:6b8:0:c22::/64     604     XXXX                XXXX    RA                      761
fola    37.9.72.32/29       2a02:6b8:0:c5d::/64     509     XXXX                XXXX    2a02:6b8:0:a10::/64     761
fola    37.9.114.0/24       2a02:6b8:0:c22::/64     604     XXXX                XXXX    RA                      761
fola    84.201.130.64/27    2a02:6b8:0:c4b::/64     1312    XXXX                XXXX    2a02:6b8:0:a13::/64     762
fola    84.201.146.80/28    2a02:6b8:0:c55::/64     614     XXXX                XXXX    2a02:6b8:0:a10::/64     761     ya.lite
fola    84.201.146.136/29   2a02:6b8:b000:1003::/64 1430    XXXX                XXXX    2a02:6b8:0:a13::/64     762     solomon
fola    84.201.146.224/28   2a02:6b8:0:c47::/64     1301    XXXX                XXXX    XXXX                    XXXX
fola    84.201.162.96/27    2a02:6b8:0:c2d::/64     593     XXXX                XXXX    XXXX                    XXXX
fola    84.201.163.0/25     2a02:6b8:0:c22::/64     604     XXXX                XXXX    RA                      761
fola    84.201.163.176/29   2a02:6b8:0:c58::/64     555     XXXX                XXXX    XXXX                    XXXX    oops
fola    87.250.246.192/26   2a02:6b8:0:c23::/64     599     XXXX                XXXX    2a02:6b8:0:a10::/64     761     balancer
fola    95.108.188.128/25   2a02:6b8:0:c1b::/64     690     XXXX                XXXX    XXXX                    XXXX
fola    95.108.189.56/29    2a02:6b8:b000:1003::/64 1430    XXXX                XXXX    2a02:6b8:0:a13::/64     762
fola    141.8.185.16/28     2a02:6b8:b000:1001::/64 1354    XXXX                XXXX    2a02:6b8:0:aa8::/64     742     KiKiMR
fola    141.8.148.0/24      2a02:6b8:0:c25::/64     603     XXXX                XXXX    2a02:6b8:f000:5005::/64     763
fola    178.154.144.0/22    2a02:6b8:0:c22::/64     604     XXXX                XXXX    RA                      761
fola    178.154.182.0/23    2a02:6b8:0:c25::/64     603     XXXX                XXXX    2a02:6b8:f000:5005::/64     763
fola    178.154.212.0/23    2a02:6b8:0:c25::/64     603     XXXX                XXXX    2a02:6b8:f000:5005::/64     763
fola    178.154.216.0/22    2a02:6b8:0:c22::/64     604     XXXX                XXXX    RA                      761
fola    178.154.220.0/24    2a02:6b8:0:c19::/64     696     XXXX                XXXX    2a02:6b8:0:a10::/64     761
fola    178.154.221.32/28   2a02:6b8:0:c31::/64     643     XXXX                XXXX    2a02:6b8:0:ab4::/64     743     SUGGEST
fola    178.154.223.0/24    2a02:6b8:0:c28::/64     604     XXXX                XXXX    RA                      761
fola    178.154.234.0/24    2a02:6b8:0:c06::/64     621     XXXX                XXXX    2a02:6b8:0:c03::/64     721     PPB
fola    178.154.239.192/26  2a02:6b8:0:c18::/64     522     XXXX                XXXX    2a02:6b8:0:a13::/64     762     CMSEARCHNETS
fola    141.8.184.0/24      2a02:6b8:0:c26::/64     595     XXXX                XXXX    2a02:6b8:f000:5005::/64 763     Spider
fola    213.180.214.0/24    2a02:6b8:0:c24::/64     604     XXXX                XXXX    RA                      761
folb    XXXX                2a02:6b8:0:f3c::/64     1388    XXXX                XXXX    2a02:6b8:0:a16::/64     768     ISS
folb    5.45.222.112/28     2a02:6b8:b000:2000::/64 1354    XXXX                XXXX    2a02:6b8:0:aa4::/64     742     KiKiMR
folb    5.45.223.0/24       2a02:6b8:0:f1f::/64     1361    XXXX                XXXX    XXXX                    XXXX    SEARCHSAND in OpenStack
folb    37.9.122.0/24       2a02:6b8:0:f33::/64     595     XXXX                XXXX    2a02:6b8:f000:5005::/64 763     Spider
folb    5.255.198.0/26      2a02:6b8:0:f1e::/64     599     XXXX                XXXX    2a02:6b8:0:a16::/64     768     balancer
folb    5.255.199.120/29    2a02:6b8:b010:6030::/64 1301    XXXX                XXXX    XXXX                    XXXX
folb    5.255.254.0/24      2a02:6b8:0:f16::/64     603     XXXX                XXXX    2a02:6b8:0:a16::/64     768     robot
folb    37.9.123.0/24       2a02:6b8:0:f2e::/64     1319    XXXX                XXXX    XXXX                    XXXX    SEARCHOPENSTACKVMNETS in OpenStack
folb    84.201.176.0/22     2a02:6b8:0:f12::/64     604     XXXX                XXXX    RA                      761
folb    84.201.180.0/25     2a02:6b8:0:f12::/64     604     XXXX                XXXX    RA                      761
folb    84.201.181.0/24     2a02:6b8:0:f1f::/64     1361    XXXX                XXXX    XXXX                    XXXX    SEARCHSAND in OpenStack
folb    84.201.182.0/23     2a02:6b8:0:f12::/64     604     XXXX                XXXX    RA                      761
folb    84.201.189.0/24     2a02:6b8:0:f14::/64     565     XXXX                XXXX    2a02:6b8:0:a93::/64     756
folb    84.201.184.224/27   2a02:6b8:0:f0a::/64     566     XXXX                XXXX    2a02:6b8:0:a93::/64     756     MR
folb    178.154.156.0/22    2a02:6b8:0:f12::/64     604     XXXX                XXXX    RA                      761
folb    178.154.160.0/23    2a02:6b8:0:f12::/64     604     XXXX                XXXX    RA                      761
folb    213.180.206.176/28  2a02:6b8:0:f29::/64     614     XXXX                XXXX    2a02:6b8:0:a16::/64     768     ya.lite
folb    213.180.207.0/24    2a02:6b8:0:f16::/64     603     XXXX                XXXX    2a02:6b8:0:a16::/64     768
sas1    XXXX                2a02:6b8:0:1a8b::/64    1388    XXXX                XXXX    2a02:6b8:0:a1a::/64     762    ISS
sas1    5.45.234.32/29      2a02:6b8:0:1a5b::/64    555     XXXX                XXXX    XXXX                    XXXX    oops
sas1    5.45.234.96/27      2a02:6b8:0:1a61::/64    565     XXXX                XXXX    2a02:6b8:0:a94::/64     866     SEARCHMRNETS
sas1    5.255.204.0/24      2a02:6b8:0:1a0e::/64    696     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    5.255.206.0/25      2a02:6b8:0:1a19::/64    522     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     CMSEARCHNETS
sas1    5.255.210.96/28     2a02:6b8:0:1a2e::/64    1301    XXXX                XXXX    XXXX                    XXXX
sas1    5.255.212.0/24      2a02:6b8:0:1a25::/64    1305    XXXX                XXXX    2a02:6b8:f030:1000::/64 865     RTMR
sas1    5.255.242.0/24      2a02:6b8:0:1a4e::/64    1324    XXXX                XXXX    2a02:6b8:f000:38::/64   705     SEARCHOPENSTACKNETS
sas1    37.9.71.0/24        2a02:6b8:0:1a11::/64    603     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    37.9.102.0/24       2a02:6b8:0:1a36::/64    690     XXXX                XXXX    XXXX                    XXXX    botanik
sas1    37.9.103.0/24       2a02:6b8:0:1a79::/64    1361    XXXX                XXXX    XXXX                    XXXX    SEARCHSAND in OpenStack
sas1    37.9.116.0/23       2a02:6b8:0:1a0e::/64    696     XXXX                XXXX    RA                      762     TenzorNet
sas1    37.9.118.32/27      2a02:6b8:0:1a7c::/64    698     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    37.9.118.128/25     2a02:6b8:0:1a37::/64    593     XXXX                XXXX    XXXX                    XXXX
sas1    37.9.125.0/24       2a02:6b8:0:1a48::/64    621     XXXX                XXXX    2a02:6b8:0:a88::/64     721     PPB
sas1    37.9.126.64/27      2a02:6b8:0:1a8a::/64    1387    XXXX                XXXX    XXXX                    767     QLOUDNETS Nirvana
sas1    37.9.127.0/24       2a02:6b8:0:1a4d::/64    1319    XXXX                XXXX    XXXX                    XXXX    OpenStack
sas1    37.140.164.64/28    2a02:6b8:b010:9a::/64   1497    XXXX                XXXX    XXXX                    XXXX     IRONIC TEST
sas1    37.140.164.80/28    2a02:6b8:0:1a0d::/64    643     XXXX                XXXX    2a02:6b8:0:ab7::/64     743    SUGGEST
sas1    37.140.165.224/27   2a02:6b8:b010:4a::/64   586     XXXX                XXXX    XXXX                    XXXX    antivirus dev
sas1    37.140.176.0/22     2a02:6b8:0:1a11::/64    603     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    37.140.181.160/30   2a02:6b8:b010:31::/64   597     XXXX                XXXX    XXXX                    XXXX    stain3
sas1    37.140.181.168/29   2a02:6b8:0:1a1c::/64    509     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     SEARCHFORMULAGPUNETS
sas1    XXXX                2a02:6b8:0:1a50::/64    510     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     build
sas1    37.140.189.64/26    2a02:6b8:0:1a1c::/64    509     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     SEARCHFORMULAGPUNETS
sas1    93.158.172.0/23     2a02:6b8:0:1a11::/64    603     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    XXXX                2a02:6b8:0:1a50::/64    510     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     build
sas1    93.158.178.64/28    2a02:6b8:0:1a7d::/64    1586    XXXX                XXXX    XXXX                    XXXX    Antivir in OpenStack
sas1    95.108.186.0/24     2a02:6b8:0:1a79::/64    1361    XXXX                XXXX    XXXX                    XXXX    SEARCHSAND in OpenStack
sas1    130.193.38.0/23     2a02:6b8:0:1a11::/64    603     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    141.8.182.0/27      2a02:6b8:0:1a63::/64    1337    XXXX                XXXX    2a02:6b8:0:a94::/64     866     SEARCHMRBACKUPNETS
sas1    141.8.182.32/27     2a02:6b8:0:1a6b::/64    1312    XXXX                XXXX    2a02:6b8:0:a1a::/64     762     webmaster
sas1    141.8.187.0/24      2a02:6b8:b000:1::/64    501     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     videorobot
sas1    178.154.187.0/24    2a02:6b8:0:1a11::/64    603     XXXX                XXXX    2a02:6b8:0:a1a::/64     762
sas1    178.154.214.0/24    2a02:6b8:0:1a19::/64    522     XXXX                XXXX    2a02:6b8:0:a1a::/64     762     CMSEARCHNETS
ash1    100.43.81.0/24      2620:10f:d00b:0::/64    595     XXXX                XXXX    2620:10f:d00f:0::/64      761     US spider nets
ash1    100.43.85.0/24      2620:10f:d00b:0::/64    595     XXXX                XXXX    2620:10f:d00f:0::/64      761     US spider nets
ash1    100.43.90.0/24      2620:10f:d00b:0::/64    595     XXXX                XXXX    2620:10f:d00f:0::/64      761     US spider nets
ash1    100.43.91.0/24      2620:10f:d00b:0::/64    595     XXXX                XXXX    2620:10f:d00f:0::/64      761     US spider nets
ash1    100.43.92.168/29    2620:10f:d00b:1::/64    696     XXXX                XXXX    2620:10f:d00f:0::/64      761     dev
ash1    199.21.99.0/24      2620:10f:d00b:0::/64    595     XXXX                XXXX    2620:10f:d00f:0::/64      761     US spider nets"""


def get_host_vlans(ip):
    """Determines which VLANs should be assigned to host with the specified IPv4 address.

    :raises InvalidHostConfiguration
    """

    network, bb6_net, bb_vlan, fb4_vlan, fb6_net, fb6_vlan = _get_network_by_ip(_get_network_configuration(), ip)

    if fb4_vlan is not None:
        raise InvalidHostConfiguration("The host has IPv4 fastbone VLAN in ya.subr ({} network).", network.exploded)

    if fb6_vlan is None:
        raise YaSubrRecordNotFoundError("There is no fastbone VLAN for {} in ya.subr.")

    return bb_vlan, fb6_vlan


def get_host_network(ip, vlan):
    """Determine which network should be used for host based on given bb IPv4 address and VLAN.
    Return IPv6Network object or None if address should not be configured via ya.subr.
    """

    network, bb6_net, bb_vlan, fb4_vlan, fb6_net, fb6_vlan = _get_network_by_ip(_get_network_configuration(), ip)

    if bb_vlan is None:
        raise InvalidHostConfiguration("There is no backbone VLAN for {} in ya.subr.", network.exploded)

    if fb6_vlan is None:
        raise InvalidHostConfiguration("There is no fastbone VLAN for {} in ya.subr.", network.exploded)

    # return backbone network
    if vlan == bb_vlan:
        if bb6_net is None:
            raise InvalidHostConfiguration("There is no backbone IPv6 network for {} in ya.subr.", network.exploded)

        return bb6_net

    # return fastbone network or None
    elif vlan == fb6_vlan:
        if fb6_net is None:
            raise InvalidHostConfiguration("There is no fastbone IPv6 network for {} in ya.subr.", network.exploded)

        if fb6_net == _RA_NETWORK:
            return None

        return fb6_net

    else:
        raise LogicalError()


def _get_network_by_ip(network_configs, ip):
    ip = IPv4Address(ip)

    matched_configs = [network_config for network_config in network_configs if ip in network_config.network]

    if not matched_configs:
        raise YaSubrRecordNotFoundError("Can't find any network for {} in ya.subr.".format(ip.exploded))

    if len(matched_configs) > 1:
        raise InvalidHostConfiguration(
            "Can't find a suitable network for {} in ya.subr. Several networks match it: {}.",
            ip.exploded,
            ", ".join(network_config.network.exploded for network_config in matched_configs),
        )

    return matched_configs[0]


_LAST_NETWORK_CONFIGS = None


def _get_network_configuration():
    global _LAST_NETWORK_CONFIGS

    try:
        network_configs = _get_network_configuration_cached()
    except Exception as e:
        if _LAST_NETWORK_CONFIGS is None:
            raise
        else:
            log.error("%s Use the cached one.", e)
            return _LAST_NETWORK_CONFIGS
    else:
        _LAST_NETWORK_CONFIGS = network_configs
        return network_configs


@lru_cache(maxsize=1)
def _get_network_configuration_cached():
    return _parse_network_configuration(_get_yasubr_networks())


def _parse_network_configuration(networks_spec):
    # Fields:
    # 0 - DC
    # 1 - IPv4 backbone network/mask
    # 2 - IPv6 backbone network/mask
    # 3 - backbone VLAN
    # 4 - IPv4 fastbone network/mask
    # 5 - IPv4 fastbone VLAN
    # 6 - IPv6 fastbone network/mask
    # 7 - IPv6 fastbone VLAN
    # 8 - Comments with spaces

    network_configs = []
    spaces_re = re.compile(r"\s+")

    for line in networks_spec.split("\n"):
        line = line.strip()
        if not line:
            continue

        network_spec = spaces_re.split(line, 8)

        try:
            if len(network_spec) < 8:
                raise ValueError

            network = network_spec[1]
            if network == "XXXX":
                continue

            network = IPv4Network(network, strict=True)

            bb6_net, bb_vlan, fb4_vlan, fb6_net, fb6_vlan = (
                _parse_v6_network(network_spec[2]),
                _parse_vlan(network_spec[3]),
                _parse_vlan(network_spec[5]),
                _parse_v6_network(network_spec[6]),
                _parse_vlan(network_spec[7]),
            )
            if bb_vlan is None:
                raise ValueError
        except ValueError:
            error = "Failed to parse network configuration from ya.subr script"
            log.error("%s: %r.", error, line)
            raise Error(error + ".")

        network_configs.append(_NetworkConfig(network, bb6_net, bb_vlan, fb4_vlan, fb6_net, fb6_vlan))

    return network_configs


def _parse_vlan(vlan):
    if vlan == "XXXX":
        return

    vlan = int(vlan)
    if vlan < walle_constants.VLAN_ID_MIN or vlan > walle_constants.VLAN_ID_MAX:
        raise ValueError

    return vlan


def _parse_v6_network(network):
    if network == "XXXX":
        return None

    if network == _RA_NETWORK:
        return network

    IPv6Network(network)
    return network


def _get_yasubr_networks():
    return NETWORKS
