"""Contains CMS maintenance logic.

Attention: We should support the case when one CMS is directly or indirectly used by several projects.
"""

import logging
import typing as tp

import gevent
from gevent.event import Event
from gevent.greenlet import Greenlet
from gevent.pool import Pool

from walle.clients.cms import CmsNamespace, CmsError
from walle.hosts import Host, HostStatus
from walle.projects import Project, DEFAULT_CMS_NAME
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)

GC_ABANDONED_CMS_TASKS_WORK_TIME = 500


def is_cms_equal_in_projects(source_project, target_project):
    source_settings = {(cms_setting.cms, cms_setting.cms_api_version) for cms_setting in source_project.cms_settings}
    target_settings = {(cms_setting.cms, cms_setting.cms_api_version) for cms_setting in target_project.cms_settings}
    return source_settings == target_settings


def is_default_cms_used_in_projects(*projects):
    for project in projects:
        for cms_setting in project.cms_settings:
            if cms_setting.cms == DEFAULT_CMS_NAME:
                return True
    return False


class CmsGc:
    def __init__(self) -> None:
        self.pool = Pool(size=10)
        self.stopped_event = Event()
        self.greenlet: tp.Optional[Greenlet] = None

    def run(self):
        """Collects abandoned CMS tasks."""

        if self.stopped_event.is_set():
            return

        self.greenlet = gevent.getcurrent()
        cms_clients = {}
        projects = set()
        for project in gevent_idle_iter(Project.objects().only("id", "cms_settings")):
            cms_client_instances = self.get_project_cms_clients(project)
            for cms_client_instance in cms_client_instances:
                cms_clients[cms_client_instance.name] = cms_client_instance
            projects.add(project.id)
        log.info("Getting active maintenance tasks...")
        results = self.pool.imap_unordered(self.get_active_maintenance, cms_clients.values())
        results = [result for result in results if result[1]]
        log.info("Got active maintenance tasks...")
        if not results:
            return

        active_hosts = self.get_active_hosts(projects)

        for cms_client, maintenance_hosts in results:
            if self.stopped_event.is_set():
                return
            self.pool.spawn(self.delete_abandoned_maintenance, cms_client, maintenance_hosts, active_hosts)

    def stop(self):
        self.stopped_event.set()
        if self.greenlet:
            self.greenlet.kill(block=False)
        self.pool.kill()

    def wait(self):
        if self.stopped_event.is_set():
            log.info("Killing gc cms pool")
            self.pool.kill()
        else:
            log.info("Waiting for gc cms pool")
            self.pool.join()

    def _get_active_cms_hosts(self, projects):
        return self._get_hosts_with_tasks(projects) | self._get_hosts_with_retained_cms_tasks(projects)

    def _get_hosts_with_retained_cms_tasks(self, projects):
        return {
            host.cms_task_id
            for host in gevent_idle_iter(
                Host.objects(cms_task_id__exists=True, project__in=projects).only("cms_task_id")
            )
        }

    def _get_hosts_with_tasks(self, projects):
        return {
            host.task.get_cms_task_id()
            for host in gevent_idle_iter(
                Host.objects(status__in=HostStatus.ALL_TASK, project__in=projects).only("task.task_id")
            )
        }

    def _get_active_cms_tasks(self, cms):
        try:
            cms_tasks = {task["id"] for task in filter(CmsNamespace.namespace_filter(), cms.get_tasks())}
        except CmsError:
            cms_tasks = None  # CMS is being nasty, hwo cares. Try next time.
        except Exception as e:
            log.exception("Failed to get a list of tasks from %s CMS: %s", cms.name, e)
            cms_tasks = None

        return cms, cms_tasks

    def _delete_abandoned_cms_tasks(self, cms, cms_task_ids, active_task_ids):
        for task_id in cms_task_ids - active_task_ids:
            log.error("Deleting abandoned %s task from %s CMS...", task_id, cms.name)

            try:
                cms.delete_task(task_id)
            except CmsError:
                pass  # CMS is being nasty, hwo cares. Try next time.
            except Exception as e:
                log.exception("Failed to delete an abandoned %s task from %s CMS: %s", task_id, cms.name, e)

    def get_project_cms_clients(self, project):
        return Project.get_cms_clients(project)

    def get_active_maintenance(self, cms_client):
        return self._get_active_cms_tasks(cms_client)

    def get_active_hosts(self, projects):
        return self._get_active_cms_hosts(projects)

    def delete_abandoned_maintenance(self, cms_client, maintenance_hosts, active_hosts):
        return self._delete_abandoned_cms_tasks(cms_client, maintenance_hosts, active_hosts)


_cms_gc = CmsGc()


def _gc_cms():
    try:
        _cms_gc.run()
    finally:
        _cms_gc.wait()
