from mongoengine import StringField, ListField

from sepelib.mongo.util import register_model
from walle.clients.cms import CmsTaskType, CmsTaskAction, CmsTaskStatus
from walle.models import Document


@register_model
class CmsTask(Document):
    id = StringField(primary_key=True, required=True, help_text="ID")
    project_id = StringField(required=True, help_text="Project ID")
    type = StringField(choices=CmsTaskType.ALL, help_text="Task type")
    issuer = StringField(help_text="Action issuer")
    action = StringField(choices=CmsTaskAction.ALL, help_text="Requested action")
    hosts = ListField(StringField(min_length=1), help_text="FQDNs", required=True)
    status = StringField(choices=CmsTaskStatus.ALL, required=True)

    api_fields = ["id", "project_id", "hosts", "status"]

    meta = {
        "collection": "cms_tasks",
        "indexes": [{"name": "unique_task", "fields": ["id", "project_id"], "unique": True}],
    }

    def __repr__(self):
        hosts = self.hosts
        len_limit = 6
        if len(hosts) > len_limit:
            str_hosts = "[" + ", ".join(hosts[: len_limit // 2] + ["..."] + hosts[-len_limit // 2 :]) + "]"
        else:
            str_hosts = "[" + ", ".join(hosts) + "]"
        return "<CmsTask id={id} project_id={project_id} type={type}, issuer={issuer}, action={action}, hosts={hosts} status={status}>".format(
            id=self.id,
            project_id=self.project_id,
            type=self.type,
            issuer=self.issuer,
            action=self.action,
            hosts=str_hosts,
            status=self.status,
        )

    def __str__(self):
        return repr(self)


@register_model
class CmsProject(Document):
    id = StringField(primary_key=True, required=True, help_text="ID")
    tasks = ListField(StringField(min_length=1), help_text="Task IDs", required=True)

    meta = {
        "collection": "cms_projects",
    }

    def __repr__(self):
        tasks = self.tasks
        len_limit = 6
        if len(tasks) > len_limit:
            str_tasks = "[" + ", ".join(tasks[: len_limit // 2] + ["..."] + tasks[-len_limit // 2 :]) + "]"
        else:
            str_tasks = "[" + ", ".join(tasks) + "]"
        return "<CmsProject id={id} tasks={tasks}>".format(id=self.id, tasks=str_tasks)

    def __str__(self):
        return repr(self)
