"""Application level constants."""

import enum

version = "5.4.871"

"""Package version.

The variable name must be lowercase because of releaselib limitations.

Major number: for major releases with huge number of changes or broken API compatibility
Minor number: for releases that require database migration or config changes
Sub-minor number: for releases that don't require migration
Sub-sub-minor number: for bugfix releases
"""


DATABASE_SCHEMA_VERSION = 40
"""Current database version."""

DEFAULT_CONFIG_PATH = "walle.conf.yaml"
"""Path to a file with default configuration."""


EMAIL_DOMAIN = "yandex-team.ru"
"""Yandex email domain."""


PROVISIONER_LUI = "lui"
"""Linux Unattended Installation (http://setup.search.yandex.net/)."""

PROVISIONER_EINE = "eine"
"""Einstellung (http://eine.yandex-team.ru/)."""

PROVISIONERS = [PROVISIONER_LUI, PROVISIONER_EINE]
"""A list of available provisioners."""

EINE_NOP_PROFILE = "nop"
FLEXY_EINE_PROFILE = "flexy"
EINE_PROFILES_WITH_DC_SUPPORT = [FLEXY_EINE_PROFILE, "common-firmware", "common-firmware-cdrom"]
"""Einstellung profile which is backed by DC engineers that solve all issues emerged during the profiling."""

EINE_IPXE_FROM_CDROM_PROFILE = "cdrom-setup"
"""Profile which boots host into iPXE from cdrom."""

VLAN_ID_MIN = 1
"""Minimum valid VLAN ID."""

VLAN_ID_MAX = 4094
"""Maximum valid VLAN ID."""

PARKING_VLAN = 999
"""All machines that are excepted from production are switched into this parking VLAN."""

PROFILING_VLAN = 542
"""VLAN which Eine uses for host profiling."""

MTN_NATIVE_VLAN = 333
"""Native MTN VLAN, actually forced by racktables when mtn-project-id is set."""

MTN_FASTBONE_VLAN = 700
"""MTN fastbone VLAN, actually forced by racktables when mtn-project-id is set."""

MTN_EXTRA_VLANS = [688, 788]
"""688 is the native VLAN for services and 788 is the fastbone for 688. These VLANs are forced by racktables too."""

SHARED_VLANS = [PARKING_VLAN, PROFILING_VLAN, MTN_NATIVE_VLAN, MTN_FASTBONE_VLAN] + MTN_EXTRA_VLANS
"""All users have a permission to switch hosts into these and from these VLANs."""


def _owner_re(template, length):
    """Create regular expression for Yandex login or Yandex Staff group by adding login restriction to the template.

    See login restriction here - https://wiki.yandex-team.ru/passport/corporate.
    """
    restriction = r"[a-zA-Z0-9_][a-zA-Z0-9_-]{{,{length}}}[a-zA-Z0-9_]".format(length=length)
    return template.format(restriction=restriction)


LOGIN_RE = _owner_re(r"^{restriction}$", length=40)
"""Regular expression for Yandex login.

See login restriction here - https://wiki.yandex-team.ru/passport/corporate.
"""

GROUP_RE = _owner_re(r"^@{restriction}$", length=100)
"""Regular expression for Yandex Staff group.

Same as LOGIN_RE but must begin with "@".
"""

OWNER_RE = r"^({}|{})$".format(_owner_re(r"{restriction}", length=38), _owner_re(r"@{restriction}", length=100))
"""Regular expression that matches both Yandex login and Yandex Staff group.

See LOGIN_RE and GROUP_RE for details.
"""

ROBOT_WALLE_OWNER = "robot-walle"
"""Username (login) of Wall-E robot, owner of all hosts."""

VLAN_SCHEME_STATIC = "static"
"""Simple VLAN scheme with static VLANs."""

VLAN_SCHEME_SEARCH = "search"
"""Yandex.Search's VLAN scheme."""

VLAN_SCHEME_CLOUD = "cloud"
"""Yandex.Cloud's VLAN scheme."""

VLAN_SCHEME_MTN = "mtn"
"""Project has MTN networking enabled. VLANs are statically configured, Wall-E doesn't switch them.
IP-address is generated based upon the "mac-address/rtc" host-id scheme (net/64 + project-id + mac-address[:32]).
"""

VLAN_SCHEME_MTN_WITHOUT_FASTBONE = "mtn-without-fastbone"
"""Same as `VLAN_SCHEME_MTN`, but without fastbone DNS records."""


VLAN_SCHEME_MTN_HOSTID = "mtn-hostid"
"""Project has MTN networking enabled. VLANs are statically configured, Wall-E doesn't switch them.
IP-address is generated based upon the "hostname" host-id scheme (net/64 + project-id + md5(hostname)[:32]).
"""

VLAN_SCHEMES = [
    VLAN_SCHEME_STATIC,
    VLAN_SCHEME_SEARCH,
    VLAN_SCHEME_MTN,
    VLAN_SCHEME_MTN_HOSTID,
    VLAN_SCHEME_CLOUD,
    VLAN_SCHEME_MTN_WITHOUT_FASTBONE,
]
"""Available VLAN schemes."""

DNS_VLAN_SCHEMES = [
    VLAN_SCHEME_STATIC,
    VLAN_SCHEME_SEARCH,
    VLAN_SCHEME_MTN,
    VLAN_SCHEME_MTN_HOSTID,
    VLAN_SCHEME_CLOUD,
    VLAN_SCHEME_MTN_WITHOUT_FASTBONE,
]
"""VLAN schemes that support DNS auto assigning."""

MTN_VLAN_SCHEMES = [VLAN_SCHEME_MTN, VLAN_SCHEME_MTN_HOSTID, VLAN_SCHEME_MTN_WITHOUT_FASTBONE]
"""VLAN schemes that work with mtn/hbf project-id."""

FASTBONE_VLAN_SCHEMES = [VLAN_SCHEME_SEARCH, VLAN_SCHEME_MTN, VLAN_SCHEME_MTN_HOSTID]
"""Vlan schemes that have exactly one fastbone."""

VLAN_SCHEMES_WITHOUT_FASTBONE = [VLAN_SCHEME_CLOUD, VLAN_SCHEME_MTN_WITHOUT_FASTBONE]
"""Vlan schemes that don't have any fastbones."""

AUTOMATED_VLAN_SCHEMES = [VLAN_SCHEME_STATIC, VLAN_SCHEME_SEARCH]
"""VLAN schemes that support VLAN switching."""

ALLOWED_VLAN_SCHEMES = [VLAN_SCHEME_STATIC, VLAN_SCHEME_SEARCH]
"""VLAN schemes that can be set from API. To enable MTN, just set hbf project id."""

UI_VLAN_SCHEMES = [
    VLAN_SCHEME_STATIC,
    VLAN_SCHEME_SEARCH,
    VLAN_SCHEME_MTN,
    VLAN_SCHEME_CLOUD,
    VLAN_SCHEME_MTN_WITHOUT_FASTBONE,
]
"""VLAN schemes that can be shown/selected in UI."""


VLAN_SCHEMES_THAT_DONT_WAIT_IN_NETWORK_LOCATION_STAGES = [
    None,
]
"""VLAN schemes that don't need _WAIT_TIMEOUT in _handle_wait_for_* stages"""

VLAN_SCHEMES_WITH_VERIFYING = [
    VLAN_SCHEME_CLOUD,
]
"""VLAN schemes that require network location verifying"""

MTN_IP_METHOD_MAC = "mac"
"""Use mac-address for host-id part of mtn IP-address."""

MTN_IP_METHOD_HOSTNAME = "hostname"
"""Use hostname for host-id part of mtn IP-address."""

MTN_IP_METHODS = [MTN_IP_METHOD_MAC, MTN_IP_METHOD_HOSTNAME]
"""All support mtn IP-address methods."""


WALLE_HOST_FQDN_SUFFIX = "wall-e.yandex.net"
"""FQDN suffix which is used for hosts controlled by Wall-E."""

DEFAULT_DNS_TTL = 3600
"""TTL for DNS records that wall-e creates."""

NETWORK_SOURCE_EINE = "eine"
NETWORK_SOURCE_LLDP = "lldp"
NETWORK_SOURCE_SNMP = "snmp"
NETWORK_SOURCE_RACKTABLES = "racktables"
NETWORK_SOURCES = [NETWORK_SOURCE_EINE, NETWORK_SOURCE_LLDP, NETWORK_SOURCE_SNMP, NETWORK_SOURCE_RACKTABLES]

MAC_SOURCE_AGENT = "agent"
MAC_SOURCE_EINE = "eine"
MAC_SOURCE_LUI = "lui"
MAC_SOURCE_SNMP = "snmp"
MAC_SOURCE_RACKTABLES = "racktables"
MAC_SOURCE_CONFIGURATION = "configuration"
MAC_SOURCES = [
    MAC_SOURCE_AGENT,
    MAC_SOURCE_EINE,
    MAC_SOURCE_LUI,
    MAC_SOURCE_SNMP,
    MAC_SOURCE_RACKTABLES,
    MAC_SOURCE_CONFIGURATION,
]


LOW_LATENCY_NETWORK_TIMEOUT = 5
"""
Timeout which is used for requests that are expected to be processed very quickly (for example when UI requires this).
"""


CMS_REJECT_RE = r"CMS.*(reject|not allowed|hasn't)"
"""Regular expression to match CMS reject"""


class SshOperation:
    FORBID = "forbid"
    """Don't try to perform operation via ssh, prefer other ways."""

    ONLY = "only"
    """Only try to perform operation via ssh, don't try other ways."""

    FALLBACK = "fallback"
    """Try to perform operation via ssh, fallback to other ways if ssh fails."""

    ALL = [FORBID, ONLY, FALLBACK]
    """All possible ssh options."""


class NetworkTarget:
    SERVICE = "service"
    """Service (eine) network (VLAN 542)"""

    PARKING = "parking"
    """Parking network (VLAN 999)"""

    PROJECT = "project"
    """Project's native vlan."""

    DEPLOY = "deploy"
    """Dynamic network: whichever set as deploy network for the host or project. Not a part of ALL."""

    DEFAULT = PROJECT
    """Current default for deploy, subject for future changes."""

    DEPLOYABLE = [SERVICE, PROJECT]
    AVAILABLE = [SERVICE, PARKING]

    ALL = [SERVICE, PARKING, PROJECT]


class YCNetworkState:
    SETUP = "setup"
    """Service network"""

    PROD = "prod"
    """Production or deploy network"""

    UNKNOWN = "unknown"
    """Not setup and not prod state"""

    L3 = "l3"
    """Interface in router mode"""

    INVALID = [UNKNOWN, L3]

    ALL = [SETUP, PROD, UNKNOWN, L3]


SUCCESS_MESSAGE = "succeeded"

PRODUCTION_ENV_NAME = "production"
TESTING_ENV_NAME = "testing"

DEFAULT_TIER_ID = 2


class TicketTrackerQueue(str, enum.Enum):
    EXP = "EXP"


# NOTE(rocco66): https://st.yandex-team.ru/WALLE-3901#61015b3848fea014f48214a0 suggestions
CREATED_FROM_WALLE_TRACKER_TAG = "created_from_walle"
INCIDENT_TICKET_TYPE = "incident"
HARDWARE_AND_FIRMWARE_EXP_COMPONENT = "Hardware & Firmware"
OTHER_EXP_VERSION = "_OTHER"


class HostType(str, enum.Enum):
    SERVER = "server"
    VM = "vm"
    MAC = "mac"
    SHADOW_SERVER = "shadow-server"

    @classmethod
    def with_manual_operation(cls) -> list[str]:
        return [cls.SERVER, cls.MAC, cls.VM]

    @classmethod
    def get_choices(cls) -> list[str]:
        return [item.value for item in cls]


HOST_TYPES_ALL = [HostType.SERVER, HostType.SHADOW_SERVER, HostType.MAC, HostType.VM]
HOST_TYPES_WITH_FULL_AUTOMATION = [HostType.SERVER]
HOST_TYPES_WITH_PARTIAL_AUTOMATION = [HostType.SERVER, HostType.MAC]
HOST_TYPES_WITH_MANUAL_OPERATION = [HostType.SERVER, HostType.MAC, HostType.VM]
HOST_TYPES_ALLOWED_WITHOUT_IPMI = [HostType.MAC, HostType.SHADOW_SERVER]
HOST_TYPES_WITH_UPDATING_LOCATION_INFO = [HostType.SERVER, HostType.MAC, HostType.SHADOW_SERVER]


class BotHostStatus(str, enum.Enum):
    OPERATION = "OPERATION"
    DIAGNOSTICS = "DIAGNOSTICS"
    RETIRED_WAITING = "RETIRED_WAITING"
    WAITING_UTIL = "WAITING_UTIL"
    RETIRED = "RETIRED"
    SETUP_ERRORS = "SETUP_ERRORS"
    SETUP_PROCESS = "SETUP_PROCESS"
    NOT_DETERMINED = "NOT_DETERMINED"
    RETURNED = "RETURNED"
    NOT_USED = "NOT_USED"
    RESERVED = "RESERVED"
    TRANSIT = "TRANSIT"
    TESTING = "TESTING"
    REPAIR_WAITING = "REPAIR_WAITING"
    REPAIR_TRANSFERRED = "REPAIR_TRANSFERRED"
    CLEAR = "CLEAR"
    SEL_WAIT = "SEL_WAIT"
    TEST_FOR_SALE = "TEST_FOR_SALE"
    XXSOLD = "XXSOLD"

    @classmethod
    def get_production_host_statuses(cls) -> list[str]:
        return [cls.OPERATION, cls.SETUP_ERRORS, cls.SETUP_PROCESS]


class CronType(str, enum.Enum):
    JUGGLER = "juggler"
    NETMON = "netmon"
    DB_CACHE_GC = "db-cache-gc"
    GC_HOSTS_WITH_MAINTENANCE_STATE_TIMEOUT = "gc-hosts-with-maintenance-state-timeout"
    GC_HOSTS_WITH_TICKET_FOR_MAINTENANCE = "gc-hosts-with-ticket-for-maintenance"
    GC_PROBATION_READY_TO_ASSIGNED = "gc-probation-ready-to-assigned"
    GC_OUTDATED_HOST_HEALTH = "gc-outdated-host-health"
    JUGGLER_DOWNTIMES_CHECK = "juggler-downtimes-check"
    GC_ADMIN_REQUESTS = "gc-admin-requests"
    DEFAULT_CMS_TASKS_SCHEDULER = "default-cms-tasks-scheduler"
    DEFAULT_CMS_PROJECT_DROP_MAINTENANCE = "default-cms-project-drop-maintenance"
    GC_ABANDONED_CMS_TASKS = "gc-abandoned-cms-tasks"
    DB_SYNC_DEVELOPMENT_STANDS = "db-sync-development-stands"
    DB_SYNC_INVENTORY = "db-sync-inventory"
    DB_SYNC_MACS = "db-sync-macs"
    DB_SYNC_RACKTABLES_NETMAP = "db-sync-racktables-netmap"
    DB_SYNC_EINE_NETMAP = "db-sync-eine-netmap"
    DB_SYNC_BOT_HARDWARE_LOCATION = "db-sync-bot-hardware-location"
    DB_SYNC_BOT_PROJECT_ID = "db-sync-bot-project-id"
    DB_SYNC_RACK_TOPOLOGY = "db-sync-rack-topology"
    SYNC_HOSTS_NETWORK_INFO = "sync-hosts-network-info"
    PROFILE_STAT_SYNC = "profile-stat-sync"
    PREORDERS_PROCESSOR = "preorders-processor"
    HOST_MACS_GC = "host-macs-gc"
    REPORT_GLOBAL_AUTOMATION = "report-global-automation"
    REPORT_PROJECT_AUTOMATION = "report-project-automation"
    REPORT_PLOT_AUTOMATION = "report-plot-automation"
    HOST_FAILURE_REPORTER = "host-failure-reporter"
    GC_REPORTS = "gc-reports"
    GC_GLOBAL_FSM_HANDBRAKE = "gc-global-fsm-handbrake"
    GC_PROJECT_FSM_HANDBRAKE = "gc-project-fsm-handbrake"
    REPORT_IDM_SYSTEM_STATUS = "report-idm-system-status"
    PREHEAT_GROUP_MEMBERS_CACHE = "preheat-group-members-cache"
    CAUTH_CERTIFICATE = "cauth-certificate"
    UPDATE_HBF_DRILLS = "update-hbf-drills"
    GC_HOSTS_STAGE_INFO = "gc-hosts-stage-info"
    SYNC_TIERS = "sync-tiers"
    SYNC_SHADOW_HOSTS = "sync-shadow-hosts"
    CONFIG_REDEFINITION_MONITORING = "config-redefinition-monitoring"
    DMC_SHARDS_PROCESSING = "dmc-shards-processing"
    INFINIBAND_INFO_SYNC = "infiniband-info-sync"


LOGICAL_DATACENTERS = ["VLX"]
