import logging
from collections import defaultdict

from walle import audit_log
from walle.authorization import ISSUER_WALLE
from walle.clients import bot
from walle.hosts import Host, HostType
from walle.projects import Project
from walle.shadow_hosts import (
    get_host_bot_info,
    get_or_create_shadow_host,
    cleanup_empty_projects_and_unlinked_maintenance_plots,
    get_or_create_project_with_maintenance_plot,
    cleanup_invalid_shadow_hosts,
)
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


EVENT_NAME = "sync-shadow-hosts"


def _sync_shadow_hosts():
    log.info("Started %s job", EVENT_NAME)
    planner_ids_to_bot_ids = _get_planner_ids_to_bot_project_ids()
    bot_ids_to_shadow_project_ids = _get_bot_project_id_to_shadow_projects_ids()
    inv_to_hosts_map = _get_inv_to_hosts_map()

    planner_id_to_new_host_bot_info_map = defaultdict(list)

    planner_id_to_existing_shadow_host_bot_info_map = defaultdict(list)

    all_shadow_hosts_inv_to_name_in_bot_map = dict()

    for raw_dict in bot.iter_hosts_info():
        host_bot_info = get_host_bot_info(raw_dict)

        if not host_bot_info.planner_id:
            log.error("Host <%s> doesn't have any planner id (abc service id). Deal with it.", host_bot_info.inv)
            #  TODO: some errors in BOT data, how to deal with it?
            continue

        #  TODO: maybe we need to process some not operation hosts???
        # if host_bot_info.oebs_status != BotHostStatus.OPERATION:
        #     continue

        existing_host = inv_to_hosts_map.get(host_bot_info.inv)

        if not existing_host:
            all_shadow_hosts_inv_to_name_in_bot_map[host_bot_info.inv] = host_bot_info.name
            planner_id_to_new_host_bot_info_map[host_bot_info.planner_id].append(host_bot_info)

        elif existing_host.type == HostType.SHADOW_SERVER:
            all_shadow_hosts_inv_to_name_in_bot_map[host_bot_info.inv] = host_bot_info.name
            bot_project_id = planner_ids_to_bot_ids[host_bot_info.planner_id]

            if existing_host.project in bot_ids_to_shadow_project_ids[bot_project_id]:
                # It's all good, host in BOT has the same ids (Bot/ABC) as host in Wall-e, don't change anything
                continue

            else:
                # Host in BOT has different ids (Bot/ABC), Wall-e need to change project id
                planner_id_to_existing_shadow_host_bot_info_map[host_bot_info.planner_id].append(host_bot_info)

    for planner_id, host_bot_infos in planner_id_to_existing_shadow_host_bot_info_map.items():
        project = get_or_create_project_with_maintenance_plot(planner_id, planner_ids_to_bot_ids, EVENT_NAME)

        for host_bot_info in host_bot_infos:
            host = inv_to_hosts_map.get(host_bot_info.inv)
            with audit_log.on_switch_project_for_host(
                ISSUER_WALLE, host.project, project.id, host.inv, host.name, host.uuid, reason=EVENT_NAME
            ):
                Host.objects(
                    inv=host_bot_info.inv, uuid=host.uuid, type=HostType.SHADOW_SERVER, project=host.project
                ).modify(set__project=project.id)
                log.info("Synced host %s to project %s", host_bot_info.inv, project.id)

    for planner_id, host_bot_infos in planner_id_to_new_host_bot_info_map.items():
        project = get_or_create_project_with_maintenance_plot(planner_id, planner_ids_to_bot_ids, EVENT_NAME)

        for host_bot_info in host_bot_infos:
            try:
                get_or_create_shadow_host(host_bot_info, project.id, reason=EVENT_NAME)
                log.info("Added host %s to project %s", host_bot_info.inv, project.id)
            except Exception as e:
                log.error("Can't add host %s to project %s: %s", host_bot_info.inv, project.id, e)

    for inv, name in all_shadow_hosts_inv_to_name_in_bot_map.items():
        try:
            host = Host.objects.only("inv", "name").get(inv=inv)
            if host.name != name:
                with audit_log.on_host_fqdn_changed(
                    ISSUER_WALLE, host.project, host.name, name, host.inv, host.uuid, EVENT_NAME
                ):
                    if not name or Host.objects(name=name).count():
                        host.modify(unset__name=True)
                    else:
                        host.modify(set__name=name)
        except Exception as e:
            log.error("Host <%s>/<%s> exists in Bot, but not in Wall-e. Deal with it. Exc: %s", inv, name, e)

    cleanup_invalid_shadow_hosts(all_shadow_hosts_inv_to_name_in_bot_map, reason=EVENT_NAME)

    cleanup_empty_projects_and_unlinked_maintenance_plots(EVENT_NAME)


def _get_planner_ids_to_bot_project_ids() -> dict[int, str]:
    result = dict()
    for bot_project_id, project in bot.get_oebs_projects().items():
        result[project["planner_id"]] = bot_project_id
    return result


def _get_bot_project_id_to_shadow_projects_ids() -> dict[str, set[str]]:
    project_collection = MongoDocument.for_model(Project)
    projects = project_collection.find(
        query={"type": HostType.SHADOW_SERVER},
        fields=(
            "id",
            "bot_project_id",
        ),
        read_preference=SECONDARY_LOCAL_DC_PREFERRED,
    )
    result = defaultdict(set)
    for project in gevent_idle_iter(projects):
        result[project.bot_project_id].add(project.id)
    return result


def _get_inv_to_hosts_map() -> dict[int, Host]:
    host_collection = MongoDocument.for_model(Host)
    hosts = host_collection.find(
        query={}, fields=("uuid", "inv", "name", "type", "project"), read_preference=SECONDARY_LOCAL_DC_PREFERRED
    )
    return {item.inv: item for item in gevent_idle_iter(hosts)}
