"""Sync tiers between projects and hosts"""

import logging

from mongoengine import Q

from sepelib.core import config
from walle.hosts import Host, HostState, HostStatus
from walle.locks import HostInterruptableLock
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

JUGGLER_MESSAGE_TEMPLATE = "Hosts with mismathed tiers count: {}"
EVENT_NAME_TEMPLATE = "{}-sync-tiers"


def _sync_tiers():
    project_collection = MongoDocument.for_model(Project)
    projects = project_collection.find(query={}, fields=("id", "tier"), read_preference=SECONDARY_LOCAL_DC_PREFERRED)

    hosts_with_mismathed_tiers_count = 0

    for project in gevent_idle_iter(projects):

        log.info("Started syncing tier for hosts in project %s.", project.id)

        host_query = (
            Q(project=project.id)
            & Q(tier__ne=project.tier)
            & Q(state__in=HostState.ALL_ASSIGNED)
            & Q(status__in=HostStatus.ALL_ASSIGNED)
        )
        for host in gevent_idle_iter(Host.objects(host_query).only("uuid", "tier")):
            with HostInterruptableLock(host.uuid, host.tier):
                prev_host_tier = host.tier
                host.modify(set__tier=project.tier)
                hosts_with_mismathed_tiers_count += 1
                log.info(
                    "Changed tier for host %s in project %s from %s to %s.",
                    host.uuid,
                    project.id,
                    prev_host_tier,
                    host.tier,
                )

        log.info("Finished syncing tier for hosts in project %s.", project.id)

    if config.get_value("tier_configuration.hosts_with_mismathed_tiers_crit_count") <= hosts_with_mismathed_tiers_count:
        log.info("Too many hosts with mismathed tiers count: %d", hosts_with_mismathed_tiers_count)
        return False
    else:
        return True
