"""Syncs BOT-Project-Id (ABC-service id) for hosts between wall-e and BOT."""

import logging
from collections import namedtuple
from itertools import islice

from mongoengine import Q

from sepelib.core import constants, config
from walle import tasks
from walle.authorization import ISSUER_WALLE
from walle.clients import bot
from walle.constants import HOST_TYPES_WITH_PARTIAL_AUTOMATION
from walle.errors import HostStateChanged
from walle.hosts import Host, HostStatus, HostState
from walle.operations_log.constants import Operation
from walle.projects import Project
from walle.stats import stats_manager, ABSOLUTE
from walle.util.gevent_tools import gevent_idle_iter, gevent_idle_generator
from walle.util.misc import StopWatch
from walle.util.mongo import MongoDocument

log = logging.getLogger(__name__)

MAX_HOSTS_TO_SYNC = 200
DEGRADE_LEVEL = 0.05

DB_SYNC_BOT_PROJECT_ID_CHECK_PERIOD = 15 * constants.MINUTE_SECONDS
_MAX_RUN = 10 * constants.MINUTE_SECONDS


class StatWr:
    def __init__(self, count_running):
        self._count_running = count_running
        self._count_started = 0
        self._count_left = None
        self._stopwatch = StopWatch()

    @staticmethod
    def set_count_total_unmatched_hosts(count):
        stats_manager.set_counter_value("bot-project-id-unmatched-total", count, aggregation=ABSOLUTE)

    @staticmethod
    def set_count_started_unmatched_hosts(count):
        stats_manager.set_counter_value("bot-project-id-unmatched-started", count, aggregation=ABSOLUTE)

    @staticmethod
    def set_count_running_unmatched_hosts(count):
        stats_manager.set_counter_value("bot-project-id-unmatched-running", count, aggregation=ABSOLUTE)

    def get_running_time(self):
        return self._stopwatch.get()

    def set_running_time(self):
        stats_manager.add_sample("bot-project-id-check-time", self._stopwatch.reset())

    def inc_started(self):
        self._count_started += 1

    def set_left_hosts(self, count):
        self._count_left = count

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self._count_left is not None:
            self.set_count_total_unmatched_hosts(self._count_left + self._count_started + self._count_running)

        self.set_count_started_unmatched_hosts(self._count_started)
        self.set_count_running_unmatched_hosts(self._count_running + self._count_started)
        self.set_running_time()


def db_sync_bot_project_id_wrapper():
    try:
        sync()
    except Exception as e:
        log.exception("Failed to sync bot project id for hosts: %s", e)
        raise


def sync():
    count_running = _get_count_of_running_hosts()
    max_hosts_to_sync = MAX_HOSTS_TO_SYNC - count_running

    with StatWr(count_running) as stats:

        if count_running < MAX_HOSTS_TO_SYNC * DEGRADE_LEVEL:
            host_to_planner_id = _get_planer_ids_for_all_hosts()
            projects = _get_projects()

            hosts_to_check = _get_hosts_to_check(projects)
            hosts_to_sync = _filter_unmatched_hosts(hosts_to_check, projects, host_to_planner_id)

            for item in gevent_idle_iter(islice(hosts_to_sync, max_hosts_to_sync)):
                try:
                    host = _fetch_host(item.host_inv)
                    tasks.schedule_bot_project_sync(
                        issuer=ISSUER_WALLE, host=host, bot_project_id=item.need_bot_project_id, reason=item.reason_str
                    )
                    stats.inc_started()
                    if stats.get_running_time() > _MAX_RUN:
                        break

                except HostStateChanged:
                    log.debug("Host %s state changed, failed to start bot syncing task", item.host_inv)

            stats.set_left_hosts(len(list(hosts_to_sync)))


def _get_projects():
    excluded_projects = config.get_value("bot_project_sync.excluded_projects")

    project_query = Q(
        # only enable for list of projects currently, see WALLE-2347
        # validate_bot_project_id=True,
        bot_project_id__exists=True,
    ) & Q(id__nin=excluded_projects)

    return {
        p.id: p.bot_project_id for p in gevent_idle_iter(Project.objects(project_query).only("id", "bot_project_id"))
    }


def _get_hosts_to_check(projects):
    hd = MongoDocument.for_model(Host)
    return hd.find(
        {
            "state": {"$in": HostState.ALL_ASSIGNED},
            "status": {"$in": HostStatus.ALL_STEADY},
            "project": {"$in": list(projects)},
            "type": {"$in": HOST_TYPES_WITH_PARTIAL_AUTOMATION},
        },
        fields=["inv", "name", "project"],
    )


def _fetch_host(host_inv):
    try:
        return Host.objects.get(inv=host_inv)
    except Host.DoesNotExist:
        raise HostStateChanged


@gevent_idle_generator
def _filter_unmatched_hosts(hosts, projects, planner_ids):
    _Result = namedtuple("Result", ["host_inv", "need_bot_project_id", "reason_str"])

    projects_map = _get_oebs_projects_mapping()
    for host in hosts:
        need_bot_project_id = projects[host.project]

        if not need_bot_project_id:
            continue

        host_bot_project_id = _planner_id_to_bot_project(
            host.inv,
            planner_ids[host.inv],
            projects_map,
        )

        if need_bot_project_id and host_bot_project_id != need_bot_project_id:
            reason_str = mk_reason(host.project, need_bot_project_id, host_bot_project_id)
            yield _Result(host.inv, need_bot_project_id, reason_str)


def _get_count_of_running_hosts():
    return Host.objects(status=Operation.BOT_PROJECT_SYNC.host_status).count()


def _get_planer_ids_for_all_hosts():
    return {host_info["inv"]: host_info.get("planner_id") for host_info in gevent_idle_iter(bot.iter_hosts_info())}


def _planner_id_to_bot_project(host_inv, planner_id, planner_to_project_map):
    if not planner_id:
        return None

    try:
        planner_id = int(planner_id)
    except ValueError as e:
        log.exception("Got an invalid planner_id for host: %s.", host_inv)

        raise bot.BotInternalError(
            "Got an invalid planner_id for host #{}: {}.".format(host_inv, e), inv=host_inv, planner_id=planner_id
        )

    return planner_to_project_map.get(planner_id, None)


def _get_oebs_projects_mapping():
    return {project["planner_id"]: project_id for project_id, project in bot.get_oebs_projects().items()}


def mk_reason(project_id, need_bot_project_id, actual_bot_project_id):
    return (
        "Project id in bot ('{actual_bot_project_id}')"
        " is different than bot-project-id in project '{project_id}' ('{need_bot_project_id}')".format(
            project_id=project_id, actual_bot_project_id=actual_bot_project_id, need_bot_project_id=need_bot_project_id
        )
    )
