"""Syncs Wall-E database with inventory."""

import logging
import operator
import typing as tp
from collections import namedtuple
from threading import Lock

from gevent.event import Event
from gevent.pool import Pool

import walle.host_status
from sepelib.core.exceptions import Error
from walle import authorization, audit_log
from walle.application import app
from walle.clients import bot
from walle.constants import HOST_TYPES_WITH_PARTIAL_AUTOMATION
from walle.errors import InvalidHostStateError
from walle.host_network import HostNetwork
from walle.host_operations import change_host_inventory_number
from walle.hosts import Host, HostState, HostStatus, sync_macs, get_host_human_id, Task, HostType
from walle.locks import lost_mongo_lock_retry
from walle.models import timestamp
from walle.operations_log.constants import Operation
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import fix_mongo_set_kwargs, merge_iterators_by_uuid
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

_MAX_CONCURRENCY = 10
_MAX_INVALID_HOSTS_DEFAULT = 250
_MAX_UPDATED_MACS_HOSTS_DEFAULT = 100
_MAX_UPDATED_INV_HOSTS_DEFAULT = 100


class BotDatabaseBroken(Error):
    """Raise when bot listings contain too many errors."""

    pass


class _Syncer:
    _InvalidateHost = namedtuple("InvalidateHost", ("host", "reason"))
    _UpdateInvHost = namedtuple("UpdateInvHost", ("host", "new_inv", "reason"))

    def __init__(self):
        self.__errors = 0
        self.__lock = Lock()

        self.__pool = Pool(_MAX_CONCURRENCY)
        self.__stopped_event = Event()

        self.__host_info_cache = {}
        self.__inventory_actualization_time = timestamp()
        self.__invalidate_hosts_list = []
        self.__update_inv_hosts_list = []
        self.__updated_macs_hosts_count = 0

        self.__invalid_hosts_limit_override = app.settings().inventory_invalid_hosts_limit or _MAX_INVALID_HOSTS_DEFAULT
        self.__updated_inv_hosts_limit_override = (
            app.settings().inventory_updated_inv_hosts_limit or _MAX_UPDATED_INV_HOSTS_DEFAULT
        )
        self.__updated_macs_hosts_limit_override = (
            app.settings().inventory_updated_macs_hosts_limit or _MAX_UPDATED_MACS_HOSTS_DEFAULT
        )

    def kill_pool(self):
        self.__stopped_event.set()
        self.__pool.kill()

    def __fetch_from_bot(self):
        self.__inv_name, self.__name_inv, self.__known_hosts_timestamp = bot.get_known_hosts()
        self.__ipmi_macs = bot.get_ipmi_macs()
        self.__macs = {host_info["inv"]: host_info["macs"] for host_info in gevent_idle_iter(bot.iter_hosts_info())}

    @staticmethod
    def __fetch_hosts_networks():
        host_fields = (
            Host.uuid.db_field,
            Host.inv.db_field,
            Host.name.db_field,
            Host.project.db_field,
            Host.state.db_field,
            Host.status.db_field,
            Host.status_audit_log_id.db_field,
            Host.rename_time.db_field,
            Host.ipmi_mac.db_field,
            Host.macs.db_field,
            Host.task.db_field + "." + Task.task_id.db_field,
            Host.tier.db_field,
        )
        host_network_fields = (
            HostNetwork.uuid.db_field,
            HostNetwork.active_mac.db_field,
            HostNetwork.active_mac_time.db_field,
            HostNetwork.active_mac_source.db_field,
        )

        host_document = MongoDocument.for_model(Host)
        host_network_document = MongoDocument.for_model(HostNetwork)

        hosts = sorted(
            gevent_idle_iter(
                host_document.find(
                    {
                        Host.state.db_field: {"$in": HostState.MAC_SYNC_STATES},
                        Host.type.db_field: {"$in": HOST_TYPES_WITH_PARTIAL_AUTOMATION},
                    },
                    host_fields,
                    read_preference=SECONDARY_LOCAL_DC_PREFERRED,
                )
            ),
            key=operator.attrgetter("uuid"),
        )

        hosts_network = sorted(
            gevent_idle_iter(
                host_network_document.find(
                    {},
                    host_network_fields,
                    read_preference=SECONDARY_LOCAL_DC_PREFERRED,
                )
            ),
            key=operator.attrgetter("uuid"),
        )

        return merge_iterators_by_uuid(iter(hosts), iter(hosts_network))

    def sync(self):
        if self.__stopped_event.is_set():
            return

        self.__fetch_from_bot()

        try:
            for host, host_network in self.__fetch_hosts_networks():
                if not host:
                    continue
                if not host_network:
                    host_network = HostNetwork.get_or_create(host.uuid)
                try:
                    self.__sync_inv_name(host)
                    self.__sync_ipmi_mac(host)
                    self.__sync_macs(host, host_network)
                except bot.BotInternalError as e:
                    log.error("Failed to sync %s with BOT database: %s", get_host_human_id(host.inv, host.name), e)
                    self.__inc_errors()

            self.__sync_returned_invalid_hosts()
            self.__invalidate_marked_hosts()
            self.__update_host_invs()
        except BotDatabaseBroken as e:
            log.error("Bot database is broken: %s", e)
            self.__inc_errors()
        else:
            self.__reset_invalid_hosts_bump()
        finally:
            lost_mongo_lock_retry(self.__pool.join)()

        return self.__errors

    def __inc_errors(self):
        with self.__lock:
            self.__errors += 1

    def __sync_inv_name(self, host):
        inv, name = host.inv, host.name
        if host.rename_time is not None and host.rename_time >= self.__known_hosts_timestamp:
            # Skip this iteration, host was renamed but new name haven't yet appeared in `known hosts` view.
            return

        try:
            name_by_inv = self.__inv_name[inv]
        except KeyError:
            # Can't find the inventory number. It may mean that host is excepted from production or it hasn't been
            # acquired by its owner yet, so we have to look deeper...
            new_inv = self.__name_inv.get(name)

            if name and new_inv:
                self.__mark_host_inv_update(
                    host, new_inv, "{} now points to #{} according to BOT.".format(name, new_inv)
                )

            elif (
                name is None
                or host.status == Operation.PREPARE.host_status
                or (host.state == HostState.MAINTENANCE and host.status == HostStatus.default(host.state))
            ):
                # The host has been added as free host without name, so it may be just a host that haven't been
                # acquired by its owner yet.

                host_info = self.__get_host_info(inv)

                if host_info is None:
                    self.__mark_host_invalid(
                        host, "There is no #{} inventory number in Bot registry.".format(inv), is_error=True
                    )
                elif bot.is_excepted_status(host_info["oebs_status"]):
                    self.__mark_host_invalid(
                        host,
                        "#{} is excepted from production: it has {} OEBS status.".format(inv, host_info["oebs_status"]),
                    )
                else:
                    # All is OK. It's probably just a host that haven't been acquired by its owner yet.
                    pass

            elif not new_inv:
                host_info = self.__get_host_info(inv)

                if host_info is None:
                    self.__mark_host_invalid(
                        host, "There is no #{} inventory number in Bot registry.".format(inv), is_error=True
                    )
                elif bot.is_excepted_status(host_info["oebs_status"]):
                    self.__mark_host_invalid(
                        host,
                        "#{} is excepted from production: it has {} OEBS status.".format(inv, host_info["oebs_status"]),
                    )
                else:
                    # do nothing.
                    log.error("Host #{} ({}) is missing from golem hardware view in bot.".format(inv, name))
        else:
            # The inventory number is OK. Checking the name...

            if (
                name is None
                or host.status in HostStatus.ALL_RENAMING
                or (host.rename_time is not None and host.rename_time > self.__known_hosts_timestamp)
            ):
                # The host has been added as free host without name or probably changing the name at this moment - don't
                # try to track name changes.
                pass
            elif name_by_inv is None:
                # The inventory number lost its name. Trying to figure out what happened...

                new_inv = self.__name_inv.get(name)
                if new_inv is None:
                    if host.state != HostState.FREE:
                        self.__mark_host_invalid(
                            host, "There is no host name associated with #{} inventory number.".format(inv)
                        )
                else:
                    self.__mark_host_invalid(
                        host,
                        "There is no host name associated with #{} inventory number. {} now points to #{}.".format(
                            inv, name, new_inv
                        ),
                    )
            elif name_by_inv != name:
                # The host has been renamed without our knowledge.
                self.__mark_host_invalid(host, "#{} is {} now (not {}).".format(inv, name_by_inv, name))

    def __sync_ipmi_mac(self, host):
        ipmi_mac = self.__get_ipmi_mac(host.inv)

        if host.ipmi_mac != ipmi_mac:
            Host.objects(inv=host.inv, state__in=HostState.MAC_SYNC_STATES).update(
                **fix_mongo_set_kwargs(set__ipmi_mac=ipmi_mac)
            )

    def __get_ipmi_mac(self, inv):
        ipmi_mac = self.__ipmi_macs.get(inv)

        if ipmi_mac is None:
            # The host either doesn't have IPMI or it's excepted from production or it hasn't been acquired by its
            # owner yet, so we have to look deeper...

            host_info = self.__get_host_info(inv)

            if host_info is not None:
                ipmi_mac = host_info.get("ipmi_mac")

        return ipmi_mac

    def __sync_macs(self, host, host_network):
        macs = self.__macs.get(host.inv)
        if macs == []:
            log.error("Got an empty MAC list for #%s from BOT.", host.inv)
            macs = None

        if sync_macs(host, host_network, macs):
            self.__updated_macs_hosts_count += 1

            if self.__updated_macs_hosts_count > self.__updated_macs_hosts_limit_override:
                raise BotDatabaseBroken(
                    "Too many 'on_active_mac_changed' events according to bot inventory lists: "
                    "{} when only up to {} is considered normal.",
                    self.__updated_macs_hosts_count,
                    self.__updated_macs_hosts_limit_override,
                )

    def __sync_returned_invalid_hosts(self):
        hosts = Host.objects(status=HostStatus.INVALID, read_preference=SECONDARY_LOCAL_DC_PREFERRED)
        for host in gevent_idle_iter(hosts):
            host_info = self.__get_host_info(host.inv)
            if (
                host_info
                and not bot.is_excepted_status(host_info["oebs_status"])
                and host.name == host_info.get("name")
            ):
                reason = "Host has been returned to production"
                walle.host_status.force_status(
                    authorization.ISSUER_WALLE,
                    host,
                    HostStatus.default(host.state),
                    only_from_current_status_id=True,
                    ignore_maintenance=True,
                    reason=reason,
                )

    def __get_host_info(self, inv):
        try:
            host_info = self.__host_info_cache[inv]
        except KeyError:
            host_info = self.__host_info_cache[inv] = bot.get_host_info(inv)

        return host_info

    def __mark_host_invalid(self, host, reason, is_error=False):
        if host.status == HostStatus.INVALID:
            return

        severity = logging.ERROR if is_error else logging.WARNING
        log.log(
            severity,
            "Invalidate %s: %s (bot cache timestamp %s, host rename time %s)",
            get_host_human_id(host.inv, host.name),
            reason,
            self.__known_hosts_timestamp,
            host.rename_time,
        )

        self.__invalidate_hosts_list.append(self._InvalidateHost(host, reason))

    def __invalidate_marked_hosts(self):
        if len(self.__invalidate_hosts_list) > self.__invalid_hosts_limit_override:
            raise BotDatabaseBroken(
                "Too many invalid hosts according to bot inventory lists: "
                "{} when only up to {} is considered normal.",
                len(self.__invalidate_hosts_list),
                self.__invalid_hosts_limit_override,
            )

        for host, reason in self.__invalidate_hosts_list:
            if self.__stopped_event.is_set():
                return
            self.__pool.spawn(self.__force_status, host, reason)

    def __force_status(self, host, reason):
        if host.task:
            log.info("Skip invalidation of %s: it has a task", get_host_human_id(host.inv, host.name))
            return  # We need to keep track of the repair tasks
        # Note: It may take a while because of host locks
        audit_entry = audit_log.on_invalidate_host(
            authorization.ISSUER_WALLE,
            host.project,
            host.inv,
            host.name,
            host.uuid,
            reason,
            scenario_id=host.scenario_id,
        )

        try:
            host = Host.objects(uuid=host.uuid).get()
            walle.host_status.force_status(
                authorization.ISSUER_WALLE,
                host,
                HostStatus.INVALID,
                only_from_current_status_id=True,
                audit_entry=audit_entry,
                ignore_maintenance=True,
                reason=reason,
            )
        except InvalidHostStateError as e:
            log.info("Failed to invalidate %s: %s", get_host_human_id(host.inv, host.name), e)
        except Exception as e:
            log.error("Failed to invalidate %s: %s", get_host_human_id(host.inv, host.name), e)
            self.__inc_errors()

    @staticmethod
    def __reset_invalid_hosts_bump():
        settings = app.settings()
        if settings.inventory_invalid_hosts_limit:
            log_entry = audit_log.on_change_global_settings(
                issuer=authorization.ISSUER_WALLE,
                new_settings={"inventory_invalid_hosts_limit": None},
                reason="synchronisation successful",
            )

            with log_entry:
                del settings.inventory_invalid_hosts_limit
                settings.save()

    def __update_host_invs(self):
        if len(self.__update_inv_hosts_list) > self.__updated_inv_hosts_limit_override:
            raise BotDatabaseBroken(
                "Too many changed inventory numbers according to bot inventory lists: "
                "{} when only up to {} is considered normal.",
                len(self.__update_inv_hosts_list),
                self.__updated_inv_hosts_limit_override,
            )

        for host, new_inv, reason in self.__update_inv_hosts_list:
            if self.__stopped_event.is_set():
                return
            ipmi_mac = self.__get_ipmi_mac(new_inv)
            self.__pool.spawn(change_host_inventory_number, host, new_inv, ipmi_mac, reason)

    def __mark_host_inv_update(self, host, new_inv, reason):
        log.info(
            "Update inventory number for %s: %s (bot cache timestamp %s, host rename time %s)",
            get_host_human_id(host.inv, host.name),
            reason,
            self.__known_hosts_timestamp,
            host.rename_time,
        )
        try:
            existing_host = Host.objects.get(inv=new_inv, type=HostType.SERVER)
            reason = "Cannot update host {} inventory number (#{} -> #{}): it is used by {}".format(
                host.name, host.inv, new_inv, existing_host.name
            )
            self.__mark_host_invalid(host, reason)
        except Host.DoesNotExist:
            self.__update_inv_hosts_list.append(self._UpdateInvHost(host, new_inv, reason))


class _SyncWrapper:
    def __init__(self):
        self.syncer: tp.Optional[_Syncer] = None

    def stop(self):
        if self.syncer:
            self.syncer.kill_pool()

    def __call__(self, *args, **kwargs):
        self.syncer = _Syncer()
        try:
            errors = self.syncer.sync()
        except Exception:
            log.exception("Failed to sync Wall-E database with inventory database:")
            return False

        log.info("Wall-E database has been synced with inventory database (%s errors).", errors)

        return errors == 0


db_sync_inventory_wrapper = _SyncWrapper()
