"""Checks host MAC addresses against BOT database and creates issues in Startrek to fix the errors."""

import io
import logging
import operator
from collections import defaultdict

from sepelib.core import config, constants
from sepelib.core.exceptions import LogicalError
from sepelib.yandex.startrek import Relationship
from walle.clients import bot, startrek
from walle.constants import HOST_TYPES_WITH_PARTIAL_AUTOMATION
from walle.host_macs import HostMacs
from walle.hosts import Host, HostStatus, get_host_human_id
from walle.models import timestamp
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import drop_none
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

LOCATION_COMPONENTS = {"AMS", "ASH", "COL", "FOL", "IVA", "MAN", "MYT", "SAS", "UGR", "VLA"}


def db_sync_macs():
    try:
        success = _check()
    except Exception as e:
        log.exception("Failed to check MAC addresses in BOT database: %s", e)
        success = False
    return success


def _check():
    bot_hosts = {}
    inv_name_mapping = {}

    ipmi_macs_invs = {}
    network_macs_invs = {}
    inv_location = {}

    log.debug("Obtaining hosts info from BOT...")

    for inv, mac in gevent_idle_iter(bot.get_ipmi_macs().items()):
        mac_invs = (inv,)
        if ipmi_macs_invs.setdefault(mac, mac_invs) is not mac_invs:
            ipmi_macs_invs[mac] += mac_invs

    for info in gevent_idle_iter(bot.iter_hosts_info()):
        inv, name, macs, oebs_status = info["inv"], info.get("name"), info["macs"], info["oebs_status"]

        if name is not None:
            inv_name_mapping[inv] = name
            if not bot.is_excepted_status(oebs_status):
                bot_hosts[name] = info

        mac_invs = (inv,)

        for mac in macs:
            if network_macs_invs.setdefault(mac, mac_invs) is not mac_invs:
                network_macs_invs[mac] += mac_invs

        for location in (info["location"].city, info["location"].datacenter, info["location"].queue):
            if location is None:
                continue

            location = location[:3].upper()
            if location in LOCATION_COMPONENTS:
                inv_location[inv] = location
                break

    log.debug("Hosts info has been successfully obtained from BOT.")

    duplicated_ipmi_macs, duplicated_network_macs, macs_intersection = _check_bot_macs(
        inv_name_mapping, ipmi_macs_invs, network_macs_invs
    )

    shared_names, shared_macs, invalid_names, invalid_hosts, invalid_macs = _check_host_macs(
        bot_hosts, inv_name_mapping, network_macs_invs
    )

    success = True

    try:
        _report_errors_to_bot(
            inv_name_mapping,
            duplicated_ipmi_macs,
            duplicated_network_macs,
            macs_intersection,
            invalid_macs,
            inv_location,
        )
    except Exception as e:
        log.exception("Failed to report about invalid MAC addresses in BOT database: %s", e)
        success = False

    try:
        _report_errors_to_admins(shared_names, shared_macs, invalid_names, invalid_hosts)
    except Exception as e:
        log.exception("Failed to report about misconfigured hosts in Wall-E cluster: %s", e)
        success = False

    return success


def _check_bot_macs(inv_name_mapping, ipmi_macs_invs, network_macs_invs):
    duplicated_ipmi_macs = []
    for mac, mac_invs in gevent_idle_iter(ipmi_macs_invs.items()):
        if len(mac_invs) != 1:
            duplicated_ipmi_macs.append((mac, mac_invs))

    duplicated_network_macs = []
    for mac, mac_invs in gevent_idle_iter(network_macs_invs.items()):
        if len(mac_invs) != 1:
            duplicated_network_macs.append((mac, mac_invs))

    macs_intersection = []
    for mac in gevent_idle_iter(set(ipmi_macs_invs).intersection(network_macs_invs)):
        macs_intersection.append((mac, ipmi_macs_invs[mac], network_macs_invs[mac]))

    return duplicated_ipmi_macs, duplicated_network_macs, macs_intersection


def _check_host_macs(bot_hosts, inv_name_mapping, network_macs_hosts):
    hosts_by_name = {}
    hosts_by_mac = {}

    host_macs_iter = HostMacs.get_collection(read_preference=SECONDARY_LOCAL_DC_PREFERRED).find(
        {HostMacs.last_time.db_field: {"$gte": timestamp() - constants.HOUR_SECONDS}},
        {
            "_id": False,
            HostMacs.name.db_field: True,
            HostMacs.macs.db_field: True,
            HostMacs.first_time.db_field: True,
            HostMacs.last_time.db_field: True,
        },
    )

    for info in gevent_idle_iter(host_macs_iter):
        name = info["name"]
        info_tuple = (info,)

        if hosts_by_name.setdefault(name, info_tuple) is not info_tuple:
            hosts_by_name[name] += info_tuple

        for mac in info["macs"]:
            if hosts_by_mac.setdefault(mac, info_tuple) is not info_tuple:
                hosts_by_mac[mac] += info_tuple

    shared_names = {}
    shared_macs = {}
    invalid_names = []
    invalid_hosts = []
    invalid_macs = []
    hosts_for_skip = []

    # Find shared names
    for name, infos in gevent_idle_iter(hosts_by_name.items()):
        if len(infos) > 1:
            # A host has been logged with different MACs

            # Time when host first appeared with new MACs
            new_host_time = max(map(operator.itemgetter("first_time"), infos))

            # Filter out body change
            infos = [info for info in infos if info["last_time"] >= new_host_time]

            if len(infos) > 1:
                shared_names[name] = [info["macs"] for info in infos]
                hosts_for_skip.append(name)
            else:
                hosts_by_name[name] = infos[0]
        else:
            hosts_by_name[name] = infos[0]
    for name in hosts_for_skip:
        del hosts_by_name[name]

    # Find shared MACs
    for mac, infos in gevent_idle_iter(hosts_by_mac.items()):
        if len(infos) <= 1:
            continue

        # A several hosts has been logged with the same MAC

        # Time when new host has appeared
        new_host_time = max(map(operator.itemgetter("first_time"), infos))

        # Filter out host renaming, body change or adding extra NICs
        infos = [info for info in infos if info["last_time"] >= new_host_time and info["name"] not in shared_names]

        if len(infos) <= 1:
            continue

        names = [info["name"] for info in infos]

        # Host name duplicates must'n be in this list: all such cases should be filtered out by shared names
        if len(names) != len(set(names)):
            raise LogicalError()

        shared_macs.setdefault(tuple(sorted(names)), []).append(mac)

        # Don't check these hosts anymore
        for name in names:
            hosts_by_name.pop(name, None)

    walle_hosts = {
        host.name
        for host in gevent_idle_iter(
            Host.objects(
                status__ne=HostStatus.INVALID,
                name__exists=True,
                type__in=HOST_TYPES_WITH_PARTIAL_AUTOMATION,
                read_preference=SECONDARY_LOCAL_DC_PREFERRED,
            ).only("name")
        )
    }

    for name, info in gevent_idle_iter(hosts_by_name.items()):
        macs = set(info["macs"])

        try:
            bot_info = bot_hosts[name]
        except KeyError:
            # Skip new records: In case of host renaming when hostname has been changed on the host but the changes
            # hasn't been committed to BOT yet.
            if timestamp() - info["first_time"] < constants.DAY_SECONDS:
                continue

            bot_names = set()

            for mac in macs:
                for inv in network_macs_hosts.get(mac, []):
                    try:
                        bot_name = inv_name_mapping[inv]
                    except KeyError:
                        pass
                    else:
                        bot_names.add(bot_name)

            # Report invalid host names only for Wall-E's hosts: it's a very widespread situation outside Wall-E when
            # host is transferred to another project only by changing its name in BOT and without erasing its disks or
            # at least changing the hostname.
            if name in walle_hosts or not bot_names.isdisjoint(walle_hosts):
                invalid_names.append(
                    {
                        "name": name,
                        "macs": macs,
                        "bot_names": bot_names,
                    }
                )

            continue

        bot_macs = set(bot_info["macs"])

        if macs.isdisjoint(bot_macs):
            invalid_hosts.append(
                {
                    "name": name,
                    "macs": macs,
                    "bot_macs": bot_macs,
                }
            )
        elif macs != bot_macs:
            invalid_host = bot_info.copy()
            invalid_host["host_macs"] = macs
            invalid_macs.append(invalid_host)

    return shared_names, shared_macs, invalid_names, invalid_hosts, invalid_macs


def _report_errors_to_bot(
    inv_name_mapping, duplicated_ipmi_macs, duplicated_network_macs, macs_intersection, invalid_macs, locations_map
):
    report = io.StringIO()
    locations = set()
    if duplicated_ipmi_macs:
        log.warning("BOT has %s duplicated IPMI MAC addresses.", len(duplicated_ipmi_macs))
        locations.update(
            _report_duplicated_macs(
                report, "Duplicated IPMI MAC addresses", duplicated_ipmi_macs, inv_name_mapping, locations_map
            )
        )

    if duplicated_network_macs:
        log.warning("BOT has %s duplicated network MAC addresses.", len(duplicated_network_macs))
        locations.update(
            _report_duplicated_macs(
                report, "Duplicated network MAC addresses", duplicated_network_macs, inv_name_mapping, locations_map
            )
        )

    if macs_intersection:
        log.warning("BOT has %s IPMI MAC addresses which intersect with network MAC addresses.", len(macs_intersection))
        _report_title(report, "IPMI MAC addresses which intersect with network MAC addresses")
        for mac, ipmi_invs, network_invs in sorted(macs_intersection, key=operator.itemgetter(0)):
            print(
                "* %%{mac}%%: IPMI: {ipmi_hosts} | Network: {network_hosts}".format(
                    mac=mac,
                    ipmi_hosts=_format_invs_list(ipmi_invs, inv_name_mapping),
                    network_hosts=_format_invs_list(network_invs, inv_name_mapping),
                ),
                file=report,
            )

            locations.update(locations_map[inv] for inv in ipmi_invs if inv in locations_map)
            locations.update(locations_map[inv] for inv in network_invs if inv in locations_map)

    if invalid_macs:
        log.warning("BOT has invalid MAC addresses for %s hosts.", len(invalid_macs))
        _report_title(report, "Invalid MAC addresses according to information from hosts")

        for info in sorted(invalid_macs, key=operator.itemgetter("name")):
            print(
                "* **{host}**: {host_macs} on host instead of {bot_macs} in BOT.".format(
                    host=info["name"],
                    host_macs=_format_mac_list(info["host_macs"]),
                    bot_macs=_format_mac_list(info["macs"]),
                ),
                file=report,
            )

            if info["inv"] in locations_map:
                locations.add(locations_map[info["inv"]])

        print("\n**Valid MAC addresses:**\n%%", file=report)
        for info in sorted(invalid_macs, key=operator.itemgetter("inv")):
            print("{}|{}".format(info["inv"], "|".join(sorted(info["host_macs"]))), file=report)
        print("%%", file=report)

    _report_to_startrek(
        summary="Invalid MAC addresses in BOT database",
        error=report.getvalue(),
        components=list(locations) or None,
        **config.get_value("inventory_check.bot_issues")
    )


def _report_errors_to_admins(shared_names, shared_macs, invalid_names, invalid_hosts):
    report = io.StringIO()

    if shared_names:
        log.warning("Found %s servers that share the same name.", len(shared_names))
        _report_title(report, "The following servers share the same name")
        for project, hosts in _group_by_project(shared_names.items(), host_names_key=operator.itemgetter(0)):
            print("* **{project}**:".format(project=project), file=report)
            for name, macs_list in sorted(hosts, key=operator.itemgetter(0)):
                print("  * **{name}**:".format(name=name), file=report)
                for macs in macs_list:
                    print("    * {macs}".format(macs=_format_mac_list(macs)), file=report)

    if shared_macs:
        log.warning("Found %s hosts that share same MAC.", len(shared_macs))
        _report_title(report, "The following hosts share the same MAC")
        for names, macs in sorted(shared_macs.items(), key=operator.itemgetter(0)):
            print(
                "* {hosts}: {macs}".format(
                    hosts=", ".join("**" + name + "**" for name in names), macs=_format_mac_list(macs)
                ),
                file=report,
            )

    if invalid_names:
        log.warning("Found %s hosts with invalid names.", len(invalid_names))
        _report_title(report, "The following hosts have invalid hostname")
        for project, hosts in _group_by_project(
            invalid_names, host_names_key=lambda host: list(host["bot_names"]) + [host["name"]]
        ):
            print("* **{project}**:".format(project=project), file=report)
            for host in sorted(hosts, key=operator.itemgetter("name")):
                print(
                    "  * **{name}** ({macs}): {suggest}.".format(
                        name=host["name"],
                        macs=_format_mac_list(host["macs"], styled=False),
                        suggest=" or ".join(host["bot_names"]) + " in BOT"
                        if host["bot_names"]
                        else "not registered in BOT",
                    ),
                    file=report,
                )

    if invalid_hosts:
        log.warning("Found %s invalid hosts.", len(invalid_hosts))
        _report_title(report, "The following hosts doesn't conform to BOT")
        for project, hosts in _group_by_project(invalid_hosts, host_names_key=operator.itemgetter("name")):
            print("* **{project}**:".format(project=project), file=report)
            for host in sorted(hosts, key=operator.itemgetter("name")):
                print(
                    "  * **{name}** has {macs} MAC with {bot_macs} specified in BOT.".format(
                        name=host["name"],
                        macs=_format_mac_list(host["macs"]),
                        bot_macs=_format_mac_list(host["bot_macs"]),
                    ),
                    file=report,
                )

    _report_to_startrek(
        summary="Misconfigured hosts in Wall-E cluster",
        error=report.getvalue(),
        **config.get_value("inventory_check.admin_issues")
    )


def _report_to_startrek(
    queue,
    summary,
    error,
    project=None,
    epic=None,
    task_type=None,
    close_transition=None,
    assignee=None,
    followers=None,
    components=None,
    fix_versions=None,
    strict_search=True,
    close_issues=True,
    dry_run=False,
):
    client = startrek.get_client()

    query = {
        "author": "me()",
        "summary": summary,
        "resolution": "empty()",
    }

    if strict_search:
        query.update(
            drop_none(
                {
                    "queue": queue,
                    "project": project,
                }
            )
        )

    issue = None
    for issue in client.get_issues(filter=query, limit=1):
        break

    if error:
        if dry_run:
            log.info("%s:\n%s", summary, error.rstrip())
        else:
            if issue is None:
                links = []
                if epic is not None:
                    links.append(
                        {
                            "relationship": Relationship.HAS_EPIC,
                            "issue": epic,
                        }
                    )

                issue = client.create_issue(
                    drop_none(
                        {
                            "type": task_type or "bug",
                            "queue": queue,
                            "project": project,
                            "components": components,
                            "fixVersions": fix_versions,
                            "summary": summary,
                            "description": error,
                            "assignee": assignee,
                            "followers": [{"login": follower} for follower in followers or []],
                            "links": links,
                        }
                    )
                )

                log.info("%s issue has been created for '%s'.", issue["key"], summary)
            elif issue["description"] != error:
                client.modify_issue(issue["key"], {"description": error})
                log.info("'%s' has been reported to %s issue.", summary, issue["key"])
            else:
                log.info("%s issue already has actual information about '%s'.", issue["key"], summary)
    elif issue is not None and close_issues:
        if dry_run:
            log.info("%s issue is going to be closed.", issue["key"])
        else:
            log.info("Closing %s issue.", issue["key"])
            client.close_issue(issue["key"], **drop_none({"transition": close_transition}))


def _report_title(report, title):
    print("\n====={title}:\n".format(title=title), file=report)


def _format_invs_list(invs, inv_name_mapping):
    return ", ".join(
        '""' + get_host_human_id(inv, inv_name_mapping.get(inv), bot_format=True) + '""' for inv in sorted(invs)
    )


def _format_mac_list(macs, styled=True):
    return ", ".join("%%" + mac + "%%" if styled else mac for mac in sorted(macs))


def _group_by_project(items, host_names_key):
    items_by_project = defaultdict(list)

    qs = Host.objects(read_preference=SECONDARY_LOCAL_DC_PREFERRED).only("project").limit(1)
    for item in items:
        host_names = host_names_key(item)
        if isinstance(host_names, str):
            host_names = [host_names]

        project = None
        for host in qs(name__in=host_names):
            project = host.project
            break

        items_by_project[project].append(item)

    for project, items in sorted(items_by_project.items(), key=operator.itemgetter(0)):
        yield ("Not in Wall-E" if project is None else project, items)


def _report_duplicated_macs(report, report_name, duplicated_macs, inv_name_mapping, locations_map):
    _report_title(report, report_name)
    locations = set()
    for mac, invs in sorted(duplicated_macs, key=operator.itemgetter(0)):
        print('* %%{mac}%%: {hosts}'.format(mac=mac, hosts=_format_invs_list(invs, inv_name_mapping)), file=report)
        locations.update(locations_map[inv] for inv in invs if inv in locations_map)

    return locations
