import logging

import pymongo
from mongoengine import Q

from sepelib.core import constants
from walle.constants import HOST_TYPES_WITH_UPDATING_LOCATION_INFO
from walle.errors import UserRecoverableError
from walle.host_network import HostNetwork
from walle.hosts import Host, HostStatus, HostLocation
from walle.models import timestamp
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import merge_iterators_by_uuid, fix_mongo_set_kwargs
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

REMOVE_TIMEOUT = constants.DAY_SECONDS
MAX_NETWORK_UPDATES = 1000
MAX_NETWORK_REMOVES = 1000


class TooManyNetworkUpdates(UserRecoverableError):
    pass


def _remove_host_network(host_network):
    time = timestamp()

    return HostNetwork.objects(
        (Q(active_mac_time__exists=False) | Q(active_mac_time__lt=time - REMOVE_TIMEOUT))
        & (Q(network_timestamp__exists=False) | Q(network_timestamp__lt=time - REMOVE_TIMEOUT))
        & (Q(ips_time__exists=False) | Q(ips_time__lt=time - REMOVE_TIMEOUT)),
        uuid=host_network.uuid,
    ).modify(remove=True)


def _merge_network_host(host, host_network):
    update = {}

    if host_network.active_mac and host_network.active_mac_source:
        update["set__active_mac"] = host_network.active_mac
        update["set__active_mac_source"] = host_network.active_mac_source

    if host_network.network_switch and host_network.network_port and host_network.network_source:
        update["set__location__switch"] = host_network.network_switch
        update["set__location__port"] = host_network.network_port
        update["set__location__network_source"] = host_network.network_source

    if host_network.ips:
        update["set__ips"] = host_network.ips

    return Host.objects(
        uuid=host.uuid,
        active_mac=host.active_mac,
        location__switch=host.location.switch,
        location__port=host.location.port,
        location__network_source=host.location.network_source,
        ips=host.ips,
    ).modify(**fix_mongo_set_kwargs(**update))


def _active_mac_not_actual(host, host_network):
    if not host_network.active_mac or not host_network.active_mac_source:
        return False
    return host.active_mac != host_network.active_mac or host.active_mac_source != host_network.active_mac_source


def _ips_not_actual(host, host_network):
    if not host_network.ips:
        return False
    return host.ips != host_network.ips


def _switch_port_not_actual(host, host_network):
    if not host_network.network_switch or not host_network.network_port or not host_network.network_source:
        return False
    return (
        host.location.switch != host_network.network_switch
        or host.location.port != host_network.network_port
        or host.location.network_source != host_network.network_source
    )


def _is_need_to_update(host, host_network):
    if host is None:
        return False
    if host.status == HostStatus.INVALID:
        return False
    return (
        _active_mac_not_actual(host, host_network)
        or _ips_not_actual(host, host_network)
        or _switch_port_not_actual(host, host_network)
    )


def _is_need_to_remove(host, host_network):
    time = timestamp()
    if host is None:
        return (
            (host_network.active_mac_time is None or host_network.active_mac_time < time - REMOVE_TIMEOUT)
            and (host_network.network_timestamp is None or host_network.network_timestamp < time - REMOVE_TIMEOUT)
            and (host_network.ips_time is None or host_network.ips_time < time - REMOVE_TIMEOUT)
        )
    return False


def __fetch_hosts_networks():
    host_fields = (
        Host.uuid.db_field,
        Host.status.db_field,
        Host.active_mac.db_field,
        Host.active_mac_source.db_field,
        Host.ips.db_field,
        Host.location.db_field + "." + HostLocation.switch.db_field,
        Host.location.db_field + "." + HostLocation.port.db_field,
        Host.location.db_field + "." + HostLocation.network_source.db_field,
    )

    host_network_fields = (
        HostNetwork.uuid.db_field,
        HostNetwork.active_mac.db_field,
        HostNetwork.active_mac_time.db_field,
        HostNetwork.active_mac_source.db_field,
        HostNetwork.ips.db_field,
        HostNetwork.ips_time.db_field,
        HostNetwork.network_switch.db_field,
        HostNetwork.network_port.db_field,
        HostNetwork.network_source.db_field,
        HostNetwork.network_timestamp.db_field,
    )

    host_document = MongoDocument.for_model(Host)
    host_network_document = MongoDocument.for_model(HostNetwork)

    hosts = list(
        gevent_idle_iter(
            gevent_idle_iter(
                host_document.find(
                    {Host.type.db_field: {"$in": HOST_TYPES_WITH_UPDATING_LOCATION_INFO}},
                    host_fields,
                    read_preference=SECONDARY_LOCAL_DC_PREFERRED,
                    sort=[(Host.uuid.db_field, pymongo.ASCENDING)],
                )
            )
        )
    )

    hosts_network = list(
        gevent_idle_iter(
            host_network_document.find(
                {},
                host_network_fields,
                read_preference=SECONDARY_LOCAL_DC_PREFERRED,
                sort=[(HostNetwork.uuid.db_field, pymongo.ASCENDING)],
            )
        )
    )

    return merge_iterators_by_uuid(iter(hosts), iter(hosts_network))


def sync_hosts_network_info_wrapper():
    try:
        update_list = []
        remove_list = []

        for host, host_network in __fetch_hosts_networks():
            if not host_network:
                host_network = HostNetwork.get_or_create(host.uuid)
            if _is_need_to_remove(host, host_network):
                remove_list.append(host_network)
            elif _is_need_to_update(host, host_network):
                update_list.append((host, host_network))
        if len(update_list) > MAX_NETWORK_UPDATES or len(remove_list) > MAX_NETWORK_REMOVES:
            raise TooManyNetworkUpdates(
                "Too many host network info updates pending: update pending: {}, remove pending: {}".format(
                    len(update_list), len(remove_list)
                )
            )
        for host, host_network in update_list:
            _merge_network_host(host, host_network)
        for host_network in remove_list:
            _remove_host_network(host_network)

    except TooManyNetworkUpdates as e:
        log.exception("Failed to sync host network info: %s", e)
        return False

    except Exception as e:
        log.exception("Failed to sync host network info: %s", e)
        raise

    log.info("Network info sync ok: updated: %d, removed: %d", len(update_list), len(remove_list))
    return True
