"""Syncs Wall-E database with bot hardware location database."""

import logging

import gevent
import natsort
from mongoengine import Q

from sepelib.core.exceptions import Error
from walle.clients import bot
from walle.constants import HOST_TYPES_WITH_UPDATING_LOCATION_INFO
from walle.hosts import Host
from walle.models import timestamp
from walle.physical_location_tree import (
    LOCATION_TREE_ID,
    LocationRack,
    LocationQueue,
    LocationDatacenter,
    LocationCity,
    LocationCountry,
    LocationTree,
    NoShortNameError,
    get_shortname,
    LocationSegment,
    LocationNamesMap,
)
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import fix_mongo_set_kwargs
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


class MissingShortNames(Error):
    def __init__(self, paths):
        messages = "; ".join(map("path {} does not have a short name".format, paths))
        super().__init__(messages)


def db_sync_bot_hardware_location_wrapper():
    try:
        _sync()
    except MissingShortNames as e:
        log.error(str(e))
        return False
    except Exception as e:
        log.exception("Failed to sync host physical location info from bot: %s", e)
        return False


def _sync_hardware(hardware_location, timestamp):
    location_lookup = LocationNamesMap.get_map()
    missing_names = []

    def is_equal_locations(object_1, object_2):
        return all(
            getattr(object_1, level) == getattr(object_2, level)
            for level in ("country", "city", "datacenter", "queue", "rack", "unit")
        )

    def is_actual_names(host_location, bot_location, logical_datacenter=None):
        return host_location.short_datacenter_name == get_shortname(
            location_lookup, bot_location, LocationSegment.DATACENTER, logical_datacenter=logical_datacenter
        ) and host_location.short_queue_name == get_shortname(
            location_lookup, bot_location, LocationSegment.QUEUE, logical_datacenter=logical_datacenter
        )

    def is_synced(host_location, bot_location, logical_datacenter=None):
        if is_equal_locations(host_location, bot_location):
            try:
                return is_actual_names(host.location, location, logical_datacenter=logical_datacenter)
            except NoShortNameError as e:
                missing_names.append(e.path)
                return True  # no means no, keep calm and carry on.
        return False

    from walle.util.mongo import MongoDocument

    hd = MongoDocument.for_model(Host)
    hosts = hd.find(
        {Host.type.db_field: {"$in": HOST_TYPES_WITH_UPDATING_LOCATION_INFO}},
        ["location", "inv", "project"],
        read_preference=SECONDARY_LOCAL_DC_PREFERRED,
    )

    logical_dcs = {
        p.id: p.logical_datacenter for p in gevent_idle_iter(Project.objects().only("id", "logical_datacenter"))
    }

    for host in gevent_idle_iter(hosts):
        try:
            logical_datacenter = logical_dcs[host.project]
            location = hardware_location[host.inv]

            if host.location is not None and is_synced(host.location, location, logical_datacenter=logical_datacenter):
                continue

            Host.objects(
                Q(location__physical_timestamp__lt=timestamp) | Q(location__physical_timestamp__exists=False),
                inv=host.inv,
            ).update(
                multi=False,
                **fix_mongo_set_kwargs(
                    set__location__country=location.country,
                    set__location__city=location.city,
                    set__location__datacenter=location.datacenter,
                    set__location__queue=location.queue,
                    set__location__rack=location.rack,
                    set__location__unit=location.unit,
                    set__location__logical_datacenter=logical_datacenter,
                    set__location__short_datacenter_name=get_shortname(
                        location_lookup, location, LocationSegment.DATACENTER, logical_datacenter=logical_datacenter
                    ),
                    set__location__short_queue_name=get_shortname(
                        location_lookup, location, LocationSegment.QUEUE, logical_datacenter=logical_datacenter
                    ),
                    set__location__physical_timestamp=timestamp,
                )
            )
        except (KeyError, AttributeError) as e:
            log.error("Can't update physical location for host %s: %s", host.inv, e)
            continue

    if missing_names:
        raise MissingShortNames(missing_names)


def build_location_tree(locations):
    tree = {}  # depth levels: country/city/datacenter/queue -> rack
    for location in gevent_idle_iter(locations.values()):
        country = location.country or "-"
        city = location.city or "-"
        datacenter = location.datacenter or "-"
        queue = location.queue or "-"
        rack = location.rack or "-"

        tree.setdefault(country, {}).setdefault(city, {}).setdefault(datacenter, {}).setdefault(queue, set())
        if rack:
            tree[country][city][datacenter][queue].add(rack)
    return tree


def _build_location_tree_models(tree, timestamp):
    def sort_naturally(iterable):
        return natsort.natsorted(iterable, key=lambda pair: pair[0])

    location_model = LocationTree(timestamp=timestamp)
    for country, cities in sort_naturally(tree.items()):
        country_model = LocationCountry(name=country)
        location_model.countries.append(country_model)
        for city, datacenters in sort_naturally(cities.items()):
            city_model = LocationCity(name=city)
            country_model.cities.append(city_model)
            for datacenter, queues in sort_naturally(datacenters.items()):

                gevent.idle()

                datacenter_model = LocationDatacenter(name=datacenter)
                city_model.datacenters.append(datacenter_model)
                for queue, racks in sort_naturally(queues.items()):
                    queue_model = LocationQueue(name=queue)
                    datacenter_model.queues.append(queue_model)
                    for rack in natsort.natsorted(racks):
                        rack_model = LocationRack(name=rack)
                        queue_model.racks.append(rack_model)
    return location_model


def _sync_location_tree(locations, timestamp):
    tree = build_location_tree(locations)

    location_model = _build_location_tree_models(tree, timestamp)
    location_model.id = LOCATION_TREE_ID

    location_model.save()


def _sync():
    when = timestamp()
    hardware_location = {
        host_info.inv: host_info.location for host_info in gevent_idle_iter(bot.get_host_location_info())
    }
    _sync_hardware(hardware_location, when)
    _sync_location_tree(bot.get_locations(), when)
