import logging
from itertools import groupby

from walle.constants import HOST_TYPES_WITH_PARTIAL_AUTOMATION
from walle.expert.rack_topology import RackTopology, save_rack_topology
from walle.hosts import Host
from walle.juggler import get_aggregate_name
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import MongoDocument

log = logging.getLogger(__name__)


def _sync():
    document = MongoDocument.for_model(Host)
    collection = Host.get_collection()
    raw_hosts = [
        document(raw_host)
        for raw_host in gevent_idle_iter(
            collection.find(
                {Host.type.db_field: {"$in": HOST_TYPES_WITH_PARTIAL_AUTOMATION}},
                ["name", "location.short_queue_name", "location.unit", "location.rack", "platform.system"],
            )
        )
    ]

    # This is for not storing all the hosts in one dict (see `groupby` behaviour)
    def group_by_key(h):
        return (str(h.location.short_queue_name), str(h.location.rack))

    raw_hosts.sort(key=group_by_key)

    for queue_rack, group in groupby(raw_hosts, group_by_key):
        queue_short_name = queue_rack[0]
        rack = queue_rack[1]
        if queue_short_name == str(None) or rack == str(None):
            continue
        aggregate = get_aggregate_name(queue_rack[0], queue_rack[1])
        aggregate_hosts = []
        for host in group:
            if host.location.unit is None:
                continue
            aggregate_hosts.append(host)
        _push_aggregate_hosts(aggregate, aggregate_hosts)


def _push_aggregate_hosts(aggregate, hosts):
    rack = RackTopology.from_hosts(aggregate, hosts)
    save_rack_topology(rack)


def db_sync_rack_topology_wrapper():
    try:
        _sync()
    except Exception as e:
        log.exception("Failed to sync hosts rack topology:", e)
        return False
