"""Syncs Wall-E database with RackTables."""

import logging

import mongoengine

import walle.clients.network.netmap
import walle.hosts
from sepelib.core import constants
from walle.constants import NETWORK_SOURCE_RACKTABLES
from walle.host_network import HostNetwork
from walle.hosts import Host, HostStatus
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED
from walle.util.net import mac_from_int

DB_SYNC_RACKTABLES_NETMAP_WORK_TIME = 90 * constants.MINUTE_SECONDS

log = logging.getLogger(__name__)


def db_sync_racktables_netmap_wrapper():
    try:
        _sync()
    except Exception as e:
        log.exception("Failed to sync host network location info from racktables: %s", e)
        return False


def _sync():
    with walle.clients.network.netmap.Netmap(several_locations_error_threshold=2) as netmap:
        hosts = list(
            Host.objects(
                macs__exists=True,
                macs__not__size=0,
                status__ne=HostStatus.INVALID,
                read_preference=SECONDARY_LOCAL_DC_PREFERRED,
            ).only("inv", "project", "name", "macs", "location")
        )

        for host in gevent_idle_iter(hosts, cpu_intensive=True):
            switch_info = netmap.get_host_switch(host.inv, host.macs, host.project, host.name)
            try:
                host_network = HostNetwork.get_or_create(host.uuid)
            except mongoengine.DoesNotExist:
                continue
            if switch_info is None:
                continue

            walle.hosts.update_network_location(
                host,
                host_network,
                switch_info.switch,
                switch_info.port,
                switch_info.timestamp,
                NETWORK_SOURCE_RACKTABLES,
            )

            walle.hosts.update_racktables_active_mac(
                host,
                host_network,
                mac_from_int(switch_info.int_mac),
                switch_info.timestamp,
            )
