"""Represents the default CMS."""

import logging

import mongoengine

from sepelib.core.exceptions import LogicalError
from walle.clients.cms import CmsTaskStatus
from walle.cms_models import CmsProject, CmsTask
from walle.errors import ResourceNotFoundError
from walle.projects import Project, DEFAULT_CMS_NAME
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)


# TODO:
# * Add authorization
# * Add basic functionality and limits


DEFAULT_MAX_BUSY_HOSTS = 5  # magic number


class NonDefaultCMSError(Exception):
    pass


class CmsProjectDoesntExist(Exception):
    pass


def get_max_busy_hosts(project):
    project_obj = Project.objects(id=project).only("cms", "cms_max_busy_hosts").get()
    if project_obj.cms != DEFAULT_CMS_NAME:
        error_message = "project '{project_id}' doesn't use default CMS".format(project_id=project)
        log.error("Default CMS: " + error_message)
        raise NonDefaultCMSError(error_message)
    return project_obj.cms_max_busy_hosts


def response(task_id, hosts, status, message=None):
    response_data = {
        "id": task_id,
        "hosts": hosts,
        "status": status,
    }
    if message:
        response_data["message"] = message
    return response_data


def add_task(project, task, dry_run=False):
    task_id = task["id"]
    seen_hosts = set()
    hosts = []
    for host in task["hosts"]:
        if host in seen_hosts:
            continue
        hosts.append(host)
        seen_hosts.add(host)

    if not hosts:
        return response(task_id, [], CmsTaskStatus.OK)

    try:
        max_busy_hosts = get_max_busy_hosts(project)
    except NonDefaultCMSError as e:
        return response(task_id, hosts, CmsTaskStatus.REJECTED, message=str(e))

    if len(hosts) > max_busy_hosts:
        return response(
            task_id,
            hosts,
            CmsTaskStatus.REJECTED,
            message="Number of hosts ({}) is greater than limit {}".format(
                len(hosts),
                max_busy_hosts,
            ),
        )

    if not dry_run:
        CmsTask(
            id=task_id,
            project_id=project,
            type=task["type"],
            issuer=task["issuer"],
            action=task["action"],
            hosts=hosts,
            status=CmsTaskStatus.IN_PROCESS,
        ).save(force_insert=True)
        CmsProject(id=project).modify(add_to_set__tasks=task_id, upsert=True)
        try:
            reschedule_tasks(project)
        except CmsProjectDoesntExist as e:
            log.error("add_task: " + str(e))
        task_status = CmsTask.objects(id=task_id, project_id=project).only("status").get().status
    else:
        task_status = CmsTaskStatus.IN_PROCESS
    return response(task_id, hosts, task_status)


def get_task(project, task_id):
    try:
        task = CmsTask.objects(id=task_id, project_id=project).get()
        return response(task_id, task.hosts, task.status)
    except mongoengine.DoesNotExist:
        raise ResourceNotFoundError("Task {} is not registered.", task_id)


def get_tasks(project):
    return [t.to_api_obj(["id", "hosts", "status"]) for t in gevent_idle_iter(CmsTask.objects(project_id=project))]


def delete_task(project, task_id):
    CmsTask.objects(project_id=project, id=task_id).delete()
    CmsProject.objects(id=project).update_one(pull__tasks=task_id)
    try:
        reschedule_tasks(project)
    except CmsProjectDoesntExist:
        pass


def default_cms_tasks_scheduler():
    cms_projects = [p.id for p in CmsProject.objects.only("id")]
    for project_id in cms_projects:
        try:
            reschedule_tasks(project_id)
        except CmsProjectDoesntExist:
            pass


def cms_maintenance_drop_stale_projects():
    projects_with_default_cms = {p.id for p in gevent_idle_iter(Project.objects().only("id"))}
    cms_projects = {p.id for p in CmsProject.objects.only("id")}
    for project_id in cms_projects | projects_with_default_cms:
        tasks = get_tasks(project_id)
        if not tasks:
            # Just in case somebody created task just a moment ago, we should not drop it,
            # hence, tasks_size=0
            CmsProject.objects(id=project_id, tasks__size=0).delete()


def drop_cms_project(project_id):
    CmsTask.objects(project_id=project_id).delete()
    CmsProject.objects(id=project_id).delete()


def reschedule_tasks(project):
    try:
        cms_project_obj = CmsProject.objects(id=project).only("tasks").get()
    except mongoengine.DoesNotExist:
        raise CmsProjectDoesntExist("Project {} doesn't exist, can't reschedule tasks for this project".format(project))

    try:
        max_busy_hosts = get_max_busy_hosts(project)
    except (NonDefaultCMSError, mongoengine.DoesNotExist):
        drop_cms_project(project)
        return

    busy_hosts = set()
    for task_id in gevent_idle_iter(cms_project_obj.tasks):
        try:
            task = CmsTask.objects(id=task_id, project_id=project).get()
        except mongoengine.DoesNotExist:
            if CmsProject.objects(id=project, tasks=task_id).update_one(pull__tasks=task_id):
                log.warning(
                    "CMS project '%s' had '%s' task that didn't exist. The task has been pulled from it.",
                    project,
                    task_id,
                )
            continue

        if task.status == CmsTaskStatus.OK:
            busy_hosts.update(task.hosts)
            continue
        elif task.status == CmsTaskStatus.IN_PROCESS:
            busy_hosts.update(task.hosts)
        else:
            raise LogicalError()

        if len(busy_hosts) > max_busy_hosts:
            break

        if not task.modify(
            {"project_id": project, "status": CmsTaskStatus.IN_PROCESS}, set__status=CmsTaskStatus.OK, upsert=False
        ):
            break
    return
