"""Checks hosts's downtime status in Juggler."""

import logging

from mongoengine import Q, DoesNotExist

from sepelib.core import constants
from walle.clients.juggler import JugglerClient, JugglerDowntimeTypes, JugglerDowntimeName
from walle.hosts import Host, HostStatus, HostState
from walle.locks import HostInterruptableLock
from walle.models import timestamp
from walle.stages import Stages
from walle.util.gevent_tools import gevent_idle_iter

_DOWNTIME_RACY_STAGES = [Stages.ALLOCATE_HOSTNAME, Stages.SET_DOWNTIME]
_ENDING_TIME_PERIOD = 3 * constants.DAY_SECONDS

log = logging.getLogger(__name__)


def _gc_jobs():
    expire_time = timestamp() + _ENDING_TIME_PERIOD
    juggler = JugglerClient()

    downtimed_hosts = juggler.get_fqdn_to_downtimes_map(suffixes=[JugglerDowntimeName.DEFAULT], only_ids=False)

    host_query = Q(on_downtime=True) | Q(name__in=list(downtimed_hosts))
    hosts = gevent_idle_iter(Host.objects(host_query).only("inv", "name", "status", "on_downtime", "state", "tier"))
    for host in hosts:
        if host.on_downtime and not _legit_downtime(host):
            log.error(
                "%s: Downtime leak: host is downtimed in %s state and %s status.",
                host.human_id(),
                host.state,
                host.status,
            )
            continue

        downtimes = downtimed_hosts.get(host.name, set())

        if host.name is not None and host.on_downtime != (host.name in downtimed_hosts):
            downtime_ids = [downtime.downtime_id for downtime in downtimes] if downtimes else None
            _fix_downtime(juggler, host, downtime_ids)

        for downtime in downtimes:
            if downtime.end_time is None or downtime.end_time <= expire_time:
                _extend_downtime(juggler, host.name, downtime)


def _fix_downtime(juggler, host, downtime_ids=None):
    log.debug("%s: Possibly invalid downtime status. Checking...", host.human_id())

    with HostInterruptableLock(host.uuid, host.tier):
        try:
            host = Host.objects(inv=host.inv, name=host.name).get()
        except DoesNotExist:
            log.debug("%s: Got a race during reloading the host info.", host.human_id())
            return

        if host.on_downtime and host.status in HostStatus.ALL_TASK and host.task.stage_name in _DOWNTIME_RACY_STAGES:
            log.debug(
                "%s: Ignore downtime inconsistency for the stage with a high risk of race condition (%s)",
                host.human_id(),
                host.task.stage_name,
            )
            return

        if host.on_downtime == juggler.is_downtimed(host.name):
            log.debug("%s: Got a race during checking host downtime status.", host.human_id())
            return

        if host.on_downtime:
            log.error("%s: Host is downtimed in Wall-E but not in Juggler.", host.human_id())
            log.warning("%s: Set downtime in Juggler.", host.human_id())

            try:
                downtime_id = juggler.set_downtime(
                    host.name,
                    "Wall-E is processing '{}' task on the host".format(host.status),
                    end_time=JugglerDowntimeTypes.DEFAULT.end_time,
                )
            except Exception as e:
                log.error("%s: Failed to set downtime in Juggler: %s", host.human_id(), e)
            else:
                log.info("%s: Successfully set downtime id %s in Juggler.", host.human_id(), downtime_id)

        else:
            log.error("%s: Host is downtimed in Juggler but it shouldn't.", host.human_id())
            log.warning("%s: Remove downtime in Juggler.", host.human_id())

            try:
                juggler.remove_downtimes(downtime_ids)
            except Exception as e:
                log.error("%s: Failed to remove downtime in Juggler: %s", host.human_id(), e)


def _extend_downtime(juggler_client, host_name, downtime):
    default_dt_params = JugglerDowntimeTypes.DEFAULT
    try:
        downtime_id = juggler_client.edit_downtime(
            host_name,
            end_time=default_dt_params.end_time,
            downtime_id=downtime.downtime_id,
            description=downtime.description,
            source=downtime.source,
        )

        log.info("%s: Successfully edit (extend) downtime id %s in Juggler.", host_name, downtime_id)
    except Exception as e:
        log.error("%s: Failed to extend downtime in Juggler: %s", host_name, e)


def _legit_downtime(host):
    return (host.state in HostState.ALL_DOWNTIME) or (host.status in HostStatus.ALL_TASK)
