"""Provides exception classes that are intended to be raised from both Wall-E internal modules and API handlers.

API helpers support these classes and will return a corresponding REST API response.
"""

import http.client

import requests

from sepelib.core.exceptions import Error


class ErrorType:
    RECOVERABLE = "recoverable"
    USER_RECOVERABLE = "user_recoverable"
    FIXABLE = "fixable"


class WalleError(Error):
    def __init__(self, message=None, *args, **kwargs):
        super().__init__(message, *args)
        self.error_type = ErrorType.FIXABLE
        self.error_context = kwargs
        self.retry_period = 100
        self.retry_limit = 0

    @staticmethod
    def is_recoverable():
        return False


class CMSMigrationError(WalleError):
    pass


class RecoverableError(WalleError):
    def __init__(self, message=None, *args, **kwargs):
        super().__init__(message, *args, **kwargs)
        self.error_type = ErrorType.RECOVERABLE
        self.retry_limit = 5

    @staticmethod
    def is_recoverable():
        return True


class UserRecoverableError(WalleError):
    def __init__(self, message=None, *args, **kwargs):
        super().__init__(message, *args, **kwargs)
        self.error_type = ErrorType.USER_RECOVERABLE


class FixableError(WalleError):
    def __init__(self, message=None, *args, **kwargs):
        super().__init__(message, *args, **kwargs)
        self.error_type = ErrorType.FIXABLE


class UnknownFixableError(WalleError):
    def __init__(self, exc, message=None, *args, **kwargs):
        super().__init__(message, *args, **kwargs)
        self.exception = exc


class HostNameTemplateError(UserRecoverableError):
    pass


class ApiError(UserRecoverableError):
    """Base class for all API errors"""

    def __init__(self, http_code, *args, **kwargs):
        super().__init__(*args, http_code=http_code, **kwargs)
        self.http_code = http_code
        self.headers = {}


class UnauthenticatedError(ApiError):
    """Raised on authentication failure."""

    def __init__(self, message, *args, **kwargs):
        super().__init__(http.client.UNAUTHORIZED, "Authentication failed: " + message, *args, **kwargs)


class UnauthorizedError(ApiError):
    """Raised on authorization failure."""

    def __init__(self, message, *args, **kwargs):
        super().__init__(http.client.FORBIDDEN, "Authorization failure: " + message, *args, **kwargs)


class BadRequestError(ApiError):
    """Generic class for most of API errors."""

    def __init__(self, *args, **kwargs):
        super().__init__(http.client.BAD_REQUEST, *args, **kwargs)


class ResourceNotFoundError(ApiError):
    """Raised when resource isn't found."""

    def __init__(self, *args, **kwargs):
        super().__init__(http.client.NOT_FOUND, *args, **kwargs)


class ResourceConflictError(ApiError):
    """Raised on any error due to request conflicting to the current resource state"""

    def __init__(self, *args, **kwargs):
        super().__init__(http.client.CONFLICT, *args, **kwargs)


class MethodNotAllowedError(ApiError):
    def __init__(self, *args, **kwargs):
        super().__init__(http.client.METHOD_NOT_ALLOWED, *args, **kwargs)


class TooManyRequestsError(ApiError):
    """Raised when user has sent too many requests in a given amount of time."""

    def __init__(self, *args, **kwargs):
        super().__init__(requests.codes.too_many_requests, "Worker is busy, try again later.", **kwargs)


# "422 Client Error: UNPROCESSABLE ENTITY for url" for now, will be better in WALLESUPPORT-1786
class UnprocessableRequest(ApiError):
    """Raised when request cannot be processed due unknown client errors."""

    def __init__(self, client_error, *args, **kwargs):
        super().__init__(
            http.client.UNPROCESSABLE_ENTITY, "UNPROCESSABLE ENTITY for client: {}".format(client_error), **kwargs
        )


class RequestValidationError(BadRequestError):
    """Raised on request validation error"""

    def __init__(self, *args, **kwargs):
        super().__init__("Request validation error: " + str(args[0]), *args[1:], **kwargs)


class HostNotFoundError(ResourceNotFoundError):
    def __init__(self, **kwargs):
        super().__init__("The specified host doesn't exist.", **kwargs)


class HostNetworkNotFoundError(ResourceNotFoundError):
    def __init__(self, **kwargs):
        super().__init__("The specified host network info doesn't exist.", **kwargs)


class InvalidHostStateError(ResourceConflictError):
    def __init__(self, host, allowed_states=None, allowed_statuses=None, forbidden_statuses=None, **kwargs):
        self.host = host
        self.allowed_states = self._to_list(allowed_states)
        self.allowed_statuses = self._to_list(allowed_statuses)
        self.forbidden_statuses = self._to_list(forbidden_statuses)
        self.message = []

        message = self._collect_error_message()
        super().__init__(message, name=host.name, inv=host.inv, state=host.state, status=host.status, **kwargs)

    @staticmethod
    def _to_list(item):
        return [item] if isinstance(item, str) else item

    def _collect_error_message(self):
        self._collect_host_info()
        self._collect_possible_transitions()
        self._collect_other_info()
        return "\n".join(self.message)

    def _collect_host_info(self):
        self.message.append(
            "The host has an invalid state for this operation.\n"
            "Host's state: {state}.\n"
            "Host's status: {status}.".format(state=self.host.state, status=self.host.status)
        )

    def _collect_possible_transitions(self):
        if self.allowed_states is not None:
            self.message.append("Allowed states: {states}.".format(states=", ".join(self.allowed_states)))

        if self.allowed_statuses is not None:
            self.message.append("Allowed statuses: {statuses}.".format(statuses=", ".join(self.allowed_statuses)))

        if self.forbidden_statuses is not None:
            self.message.append("Forbidden statuses: {statuses}.".format(statuses=", ".join(self.forbidden_statuses)))

        if (
            self.allowed_statuses
            and self.host.status not in self.allowed_statuses
            or self.forbidden_statuses
            and self.host.status in self.forbidden_statuses
        ):
            self.message.append("You can try to change host's status to allowed one.")

    def _collect_other_info(self):
        # To void ImportError
        from walle.hosts import HostStatus

        if self.host.status == HostStatus.INVALID:
            self.message.append(
                "For next actions you can look here: https://docs.yandex-team.ru/wall-e/faq#faq-invalid."
            )


class OperationRestrictedError(ResourceConflictError):
    def __init__(self, *args, **kwargs):
        if not args:
            args = ["Operation restricted for this host."]

        super().__init__(*args, **kwargs)


class HostUnderMaintenanceError(OperationRestrictedError):
    def __init__(self, owner, **kwargs):
        super().__init__(
            "The host is under maintenance by {}. "
            "Add 'ignore maintenance' flag to your request if this action won't break anything.",
            owner,
            owner=owner,
            **kwargs
        )


class OperationRestrictedForHostTypeError(OperationRestrictedError):
    def __init__(self, obj_type: str, allowed_types: list[str], **kwargs):
        super(OperationRestrictedError, self).__init__(
            "Operation allowed for types '{}'. Object has type '{}'.", ", ".join(allowed_types), obj_type, **kwargs
        )


class ResourceAlreadyExistsError(ResourceConflictError):
    """Raised when someone POSTs a resource that already exists"""


class InvalidHostNameError(BadRequestError):
    def __init__(self, name, error=None, **kwargs):
        if error is None:
            super().__init__("Invalid host name: {}.", name, name=name, **kwargs)
        else:
            super().__init__("Invalid host name {}: {}", name, error, name=name, error=error, **kwargs)


class InvalidProfileNameError(BadRequestError):
    def __init__(self, profile, **kwargs):
        super().__init__("Invalid Einstellung profile name: {}.", profile, profile=profile, **kwargs)


class DNSDomainNotAllowedInCertificator(BadRequestError):
    def __init__(self, dns_domain, **kwargs):
        err_msg = "DNS domain {} is not in certficator white list.".format(dns_domain)
        super().__init__(err_msg, dns_domain=dns_domain, **kwargs)


class DNSAutomationValidationError(BadRequestError):
    def __init__(self, invalid_field, error, **kwargs):
        err_msg = "Field '{}' {}".format(invalid_field, error)
        super().__init__(err_msg, invalid_field=invalid_field, error=error, **kwargs)


class InconsistentBotProjectId(BadRequestError):
    pass


class UserNotInOwningABCService(UnauthorizedError):
    def __init__(self, user, abc_service_slug, bot_project_id, **kwargs):
        super().__init__(
            "User {} does not belong to ABC service {} that manages BOT project {}",
            user,
            abc_service_slug,
            bot_project_id,
            user=user,
            abc_service=abc_service_slug,
            bot_project_id=bot_project_id,
            **kwargs
        )


class NoInformationError(BadRequestError):
    pass


class InvalidDeployConfiguration(BadRequestError):
    pass


class InvalidHostConfiguration(BadRequestError):
    pass


class YaSubrRecordNotFoundError(InvalidHostConfiguration):
    """Raised in case when ya.subr don't have records about adress."""

    pass


class StateChanged(Exception):
    """Raised in case when something have changed in the way that does not allow us to proceed with the task."""

    pass


class HostStateChanged(StateChanged):
    """Raised in case when other instance have already processed host's health state."""

    pass


class NoSchemaForCheckError(ApiError):
    def __init__(self, **kwargs):
        super().__init__(http.client.INTERNAL_SERVER_ERROR, "No schema for given status and type.", **kwargs)


class BotEmptyNameError(ApiError):
    """Raised on adding host without name in bot at any state besides free"""

    def __init__(self, *args, **kwargs):
        super().__init__(http.client.CONFLICT, *args, **kwargs)
