import logging

import walle.hosts
from sepelib.core.exceptions import Error, LogicalError
from walle import audit_log, authorization, projects
from walle.application import app
from walle.errors import ResourceConflictError
from walle.expert import failure_log, automation_plot
from walle.expert.automation.limits import (
    GlobalLimitExceeded,
    ProjectLimitExceeded,
    global_limits,
    project_limits,
    plot_limits,
    PlotLimitExceeded,
)
from walle.expert.types import get_limit_name
from walle.locks import HostHealingInterruptableGlobalLock
from walle.util.misc import StopWatch, DummyContextManager

log = logging.getLogger(__name__)


class AutomationDisabledError(Error):
    """Raised when an automated action is restricted due to disabled automation."""


class AutomationDisabledGloballyError(AutomationDisabledError):
    """Automation is disabled globally for all project."""


class AutomationDisabledForProjectError(AutomationDisabledError):
    """Automation is disabled for the project."""


class CheckDisabledError(AutomationDisabledError):
    """Check that triggered failure is disabled."""


class AutomationContext:
    def __init__(self, project_id, global_automation, project_automation_context, automation_plot=None):
        """

        :type project_id: str
        :type global_automation: global_automation.HealingAutomation | global_automation.DnsAutomation
        :type project_automation_context: project_automation.HealingAutomation | project_automation.DnsAutomation
        :type automation_plot: AutomationPlot
        """
        self.project_id = project_id

        self.global_automation = global_automation
        self.project_automation = project_automation_context
        self.automation_plot = automation_plot

        self.settings = app.settings()

    def check_automation_settings(self, project_obj=None):
        if not self.global_automation_enabled():
            raise AutomationDisabledGloballyError(
                "{} is disabled for all projects.".format(self.global_automation.get_automation_label())
            )

        if not self.project_automation_enabled(project_obj):
            raise AutomationDisabledForProjectError(
                "{} is disabled for {} project.", self.project_automation.get_automation_label(), self.project_id
            )

    def global_automation_enabled(self):
        return self.global_automation.is_enabled(self.settings)

    def project_automation_enabled(self, project_obj=None):
        if project_obj:
            return self.project_automation.enabled_for_project(project_obj)
        else:
            return self.project_automation.enabled_for_project_id(self.project_id)

    def register_automated_failure(self, host, decision, automatic_healing=False):
        with StopWatch("Global Heal Lock. Check global automation settings"):
            self.check_automation_settings()

        if automatic_healing:
            # NOTE(rocco66): remove this lock?
            global_heal_lock = HostHealingInterruptableGlobalLock()
        else:
            global_heal_lock = DummyContextManager()

        with global_heal_lock:
            with StopWatch("Global Heal Lock. Register failure"):
                if not self.register_failure(host, decision):
                    # no failures
                    return

            if host.task and host.task.enable_auto_healing:
                # do not check limits for manual tasks, manual tasks may run in big batches and
                # are expected to heal hosts even when automation is disabled.
                return

            try:
                with StopWatch("Global Heal Lock. Check limits"):
                    self._check_automation_limits(decision)

            except GlobalLimitExceeded as e:
                self._disable_global_automation(host, decision, str(e))

            except ProjectLimitExceeded as e:
                self._disable_project_automation(host, decision, str(e))

            except PlotLimitExceeded as e:
                if self.automation_plot and self.automation_plot.have_check(e.failure):
                    self._disable_check(e.failure, host, decision, reason=str(e))
                else:
                    raise LogicalError()

    def register_failure(self, host, decision):
        # this method just registers failure, does not check limits and disables automation.
        # see `register_automated_failure` for the one that does all that

        failures = decision.failures
        if not failures:
            # Decisions without failures usually generated by FSM to upgrade task to some other action.
            return False

        # No locks, no atomic transactions: if we hit the limit, we stop processing _all_ failures,
        # not just those unfortunate that got late.
        not_count = host.state in walle.hosts.HostState.ALL_IGNORED_LIMITS_COUNTING
        credited = None
        if not not_count:
            credited = self._acquire_credit(decision)

        failure_log.register_failure(host, failures, credited, not_count)

        return True

    def _failure_processing_cancelled_log(self, host, decision, reason):
        error = "Action {} not taken: {}".format(decision.action, reason)

        return audit_log.on_failure_processing_cancelled(
            self.project_id, host.inv, host.name, host.uuid, decision.reason, error
        )

    def _automation_plot_update_log(self, check_name, reason):
        return audit_log.on_update_automation_plot(
            authorization.ISSUER_WALLE, self.automation_plot.id, {check_name: "disabled"}, reason
        )

    def _disable_global_automation(self, host, decision, reason):
        with self._failure_processing_cancelled_log(host, decision, reason):
            self.global_automation.disable(reason)

        raise AutomationDisabledGloballyError(reason)

    def _disable_project_automation(self, host, decision, reason):
        with self._failure_processing_cancelled_log(host, decision, reason):
            try:
                self.project_automation.disable_automation(authorization.ISSUER_WALLE, self.project_id, reason=reason)
            except projects.ProjectNotFoundError as e:
                log.critical("Failed to disable automation for '%s' project: %s", self.project_id, e)

        raise AutomationDisabledForProjectError(reason)

    def _disable_check(self, check_name, host, decision, reason):
        with self._failure_processing_cancelled_log(host, decision, reason):
            try:
                automation_plot.disable_check_in_plot(
                    authorization.ISSUER_WALLE, self.automation_plot, check_name, reason=reason
                )
            except ResourceConflictError as e:
                log.warn(
                    "Failed to disable check '%s' in automation plot '%s': %s",
                    self.automation_plot.id,
                    check_name,
                    str(e),
                )

        raise CheckDisabledError(reason)

    def _acquire_credit(self, decision):
        credited_failures = []
        for failure in decision.failures:
            if self.project_automation.acquire_credit(self.project_id, get_limit_name(failure)):
                credited_failures.append(failure)

        return credited_failures

    def _check_automation_limits(self, decision):
        limit_checkers = [global_limits(self.settings), project_limits(self.project_id, self.project_automation)]

        if self.automation_plot:
            limit_checkers.append(plot_limits(self.project_id, self.project_automation, self.automation_plot))

        for check_limits in limit_checkers:
            check_limits(decision)
