"""Global Wall-E automation management."""

import logging

from cachetools.func import ttl_cache

from sepelib.core import config
from sepelib.core.exceptions import LogicalError
from walle import audit_log, authorization
from walle.application import app
from walle.errors import ResourceConflictError
from walle.models import Settings, timestamp
from walle.util import notifications

log = logging.getLogger(__name__)


class _BaseAutomation:
    def __init__(self, field, label):
        self._field = field
        self._label = label

    def is_enabled(self, settings=None):
        if settings is None:
            settings = app.settings()

        enabled = self._enabled_in_config() and not self._disabled_in_settings(settings)
        return enabled

    @ttl_cache(maxsize=1, ttl=5)
    def is_enabled_cached(self):
        return self.is_enabled()

    def disable(self, reason, issuer=None):
        settings = app.settings()
        if not self._disabled_in_settings(settings):
            self._save_settings(settings, disable=True, issuer=issuer, reason=reason)
        self.is_enabled_cached.cache_clear()

    def enable(self, reason, issuer=None):
        # TODO credis here
        settings = app.settings()
        if not self._disabled_in_settings(settings):
            return

        if not self._enabled_in_config():
            raise ResourceConflictError("Rejecting to enable {}: it is disabled in config.".format(self._label))

        self._save_settings(settings, disable=False, issuer=issuer, reason=reason)
        self.is_enabled_cached.cache_clear()

    def get_automation_label(self):
        return self._label

    def _disabled_in_settings(self, settings):
        """Get automation status from settings."""
        return getattr(settings, self._field)

    def _save_settings(self, settings, disable, issuer=None, reason=None):
        if issuer is None:
            issuer = authorization.ISSUER_WALLE

        update = {"set__" + self._field: disable}

        if disable:
            # use str.format() intentionally to split up dns automation from healing automation notifications in sentry.
            log.critical("%s. Disable all {}.".format(self.get_automation_label()), reason)
        else:
            # use str.format() intentionally to split up dns automation from healing automation notifications in sentry.
            log.warn("Enabling global {} (%s).".format(self.get_automation_label()), reason)
            update.update(set__failure_log_start_time=timestamp())

        with self._audit_log(issuer, enable=not disable, reason=reason):
            Settings.objects(id=settings.id).update_one(**update)
            if disable:
                self._on_disabled(issuer, reason=reason)
            else:
                self._on_enabled(issuer, reason=reason)

    def _enabled_in_config(self):
        """Check automation type enabled in config. Implemented in child class."""
        raise LogicalError

    def _audit_log(self, issuer, enable, reason):
        """Create audit log entry. Implemented in child class."""
        raise LogicalError

    def _on_enabled(self, issuer, reason):
        """Send notification when automation type enabled. Implemented in child class."""
        raise LogicalError

    def _on_disabled(self, issuer, reason):
        """Send notification when automation type disabled. Implemented in child class."""
        raise LogicalError


class HealingAutomation(_BaseAutomation):
    def _enabled_in_config(self):
        return config.get_value("automation.enabled")

    def _audit_log(self, issuer, enable, reason):
        return audit_log.on_change_healing_status(issuer, enable=enable, reason=reason)

    def _on_enabled(self, issuer, reason):
        return notifications.on_healing_automation_enabled(issuer, reason=reason)

    def _on_disabled(self, issuer, reason):
        return notifications.on_healing_automation_disabled(issuer, reason=reason)


class DnsAutomation(_BaseAutomation):
    def _enabled_in_config(self):
        return config.get_value("automation.enabled") and config.get_value("dns_fixer.enabled")

    def _audit_log(self, issuer, enable, reason):
        return audit_log.on_change_dns_automation_status(issuer, enable=enable, reason=reason)

    def _on_enabled(self, issuer, reason):
        return notifications.on_dns_automation_enabled(issuer, reason=reason)

    def _on_disabled(self, issuer, reason):
        return notifications.on_dns_automation_disabled(issuer, reason=reason)


GLOBAL_HEALING_AUTOMATION = HealingAutomation(
    field="disable_healing_automation",
    label="automated healing",
)

GLOBAL_DNS_AUTOMATION = DnsAutomation(
    field="disable_dns_automation",
    label="DNS automation",
)
