"""Project's checks configuration management."""

from mongoengine import fields, document

from sepelib.core.exceptions import LogicalError
from sepelib.mongo.util import register_model
from walle import constants as walle_constants, audit_log
from walle.authorization import blackbox, has_iam
from walle.errors import ResourceNotFoundError, ResourceConflictError
from walle.models import Document, timestamp
from walle.util import notifications, cache

AUTOMATION_PLOT_BASIC_ID = "__basic"
AUTOMATION_PLOT_FULL_FEATURED_ID = "wall-e-hw-checks"

AUTOMATION_PLOT_ID_REGEX = r"(^[a-z][a-z0-9]*(?:-[a-z0-9]+)*$|" + AUTOMATION_PLOT_FULL_FEATURED_ID + ")"

_CHECK_NAME_REGEX = r"^[a-zA-Z][a-zA-Z0-9_.]*(?:-[a-zA-Z0-9_.]+)*$"


AUTOMATION_PLOTS_CHECKS_VALUE_TIMEOUT = 60
AUTOMATION_PLOTS_CHECKS_ERROR_TIMEOUT = 15
"""get_all_automation_plots_checks() cache timeouts control"""


class AutomationPlotNotFoundError(ResourceNotFoundError):
    def __init__(self):
        super().__init__("Specified Automation Plot doesn't exist.")


class Limit(document.EmbeddedDocument):
    period = fields.StringField(required=True, help_text="Period for limit")
    limit = fields.IntField(required=True, min_value=0, help_text="Limit for check failures during given period")


class Check(document.EmbeddedDocument):
    name = fields.StringField(required=True, regex=_CHECK_NAME_REGEX, max_length=32, help_text="Check name")
    enabled = fields.BooleanField(default=True, required=True, help_text="Check toggle")
    reboot = fields.BooleanField(default=None, help_text="Wall-E should try to fix problem with reboot")
    profile = fields.BooleanField(default=None, help_text="Wall-E should try to fix problem with profile")
    redeploy = fields.BooleanField(default=None, help_text="Wall-E should try to fix problem with redeploy")
    wait = fields.BooleanField(default=None, help="Wall-E should wait for this check to become passed")
    report_failure = fields.BooleanField(default=True, help_text="Wall-E should try to report about problem")
    limits = fields.EmbeddedDocumentListField(Limit, required=False, default=None)
    start_time = fields.LongField(required=True, default=timestamp, help_text="Cut-off timestamp for limits")


@register_model
class AutomationPlot(Document):
    id = fields.StringField(
        primary_key=True, required=True, regex=AUTOMATION_PLOT_ID_REGEX, max_length=32, help_text="ID"
    )

    # Don't allow long names now to not confuse UI
    name = fields.StringField(min_length=1, max_length=32, required=True, unique=True, help_text="Name")

    yc_iam_folder_id = fields.StringField(help_text="Binded YC floder ID for IAM")

    # Can't make it required because MongoEngine forces required fields to be non-empty
    owners = fields.ListField(fields.StringField(regex=walle_constants.OWNER_RE), help_text="Owners")

    checks = fields.ListField(fields.EmbeddedDocumentField(Check), default=list, help_text="List of checks with rules")

    meta = {
        "collection": "automation_plot",
    }

    default_api_fields = ("id", "name")
    api_fields = default_api_fields + ("owners", "checks", "yc_iam_folder_id")

    @classmethod
    def with_fields(cls, fields):
        return cls.objects.only(*cls.api_query_fields(fields or None))

    def have_check(self, check_name):
        return any(check.name == check_name for check in self.checks)

    def enable_check(self, check_name):
        return self.modify(
            query=dict(checks__name=check_name), set__checks__S__enabled=True, set__checks__S__start_time=timestamp()
        )

    def disable_check(self, check_name):
        return self.modify(
            query=dict(checks__name=check_name),
            set__checks__S__enabled=False,
        )

    def authorize(self, issuer):
        if has_iam():
            return

        # Protects us from using an object without required fields
        if self.id is None or self.owners is None:
            raise LogicalError()

        error_message = "You must be '{}' automation plot owner to perform this request.".format(self.id)
        blackbox.authorize(issuer, error_message, owners=self.owners, authorize_admins=True)


def get_automation_plot(plot_id, fields=None):
    try:
        return AutomationPlot.with_fields(fields).get(id=plot_id)
    except AutomationPlot.DoesNotExist:
        raise AutomationPlotNotFoundError()


def get_all_automation_plots(fields):
    return AutomationPlot.with_fields(fields)


def authorize(issuer, plot_id, fields=None):
    fields = ("id", "owners") + tuple(fields or ())

    plot = get_automation_plot(plot_id, fields)
    plot.authorize(issuer)

    return plot


def enable_check_in_automation_plot(issuer, automation_plot, check_name, reason):
    plot_id = automation_plot.id

    with audit_log.on_update_automation_plot(issuer, plot_id, {check_name: "enabled"}, reason):
        if not automation_plot.enable_check(check_name):
            raise ResourceConflictError("Automation plot {} does not have check {}", plot_id, check_name)

    notifications.on_automation_plot_check_enabled(issuer, plot_id, check_name, reason=reason)


def disable_check_in_plot(issuer, automation_plot, check_name, reason):
    plot_id = automation_plot.id

    with audit_log.on_update_automation_plot(issuer, plot_id, {check_name: "disabled"}, reason):
        if not automation_plot.disable_check(check_name):
            raise ResourceConflictError("Automation plot {} does not have check {}.", plot_id, check_name)

    notifications.on_automation_plot_check_disabled(issuer, plot_id, check_name, reason=reason)


@cache.cached_with_exceptions(
    value_ttl=AUTOMATION_PLOTS_CHECKS_VALUE_TIMEOUT, error_ttl=AUTOMATION_PLOTS_CHECKS_ERROR_TIMEOUT
)
def get_all_automation_plots_checks():
    """
    Get list of all unique checks added to all automation plots.

    Avoid mocking automation plots in database. Mock this method instead and @cached won't hurt you.
    :return: Set[str]
    """
    checks = set()
    plots = AutomationPlot.objects().only("checks")

    # use loop instead set generator for readability
    for plot in plots:
        checks.update(c.name for c in plot.checks)

    return checks
