from sepelib.core import config
from walle.expert import constants
from walle.expert.constants import (
    UNREACH_PERIOD,
    SSH_CHECK_PERIOD,
    PASSIVE_CHECK_PERIOD,
    NETMON_POLLING_PERIOD,
    UNREACH_ACCURACY,
    SSH_CHECK_ACCURACY,
    HW_WATCHER_CHECK_ACCURACY,
    PASSIVE_CHECK_ACCURACY,
    NETMON_CHECK_ACCURACY,
)
from walle.expert.types import CheckType
from walle.util.limits import parse_period


def get_check_period(check_type):
    if check_type == CheckType.UNREACHABLE:
        return UNREACH_PERIOD
    if check_type == CheckType.SSH:
        return SSH_CHECK_PERIOD
    if check_type in CheckType.ALL_PASSIVE:
        return PASSIVE_CHECK_PERIOD
    if check_type == CheckType.WALLE_RACK:
        return UNREACH_PERIOD
    if check_type == CheckType.WALLE_RACK_OVERHEAT:
        return PASSIVE_CHECK_PERIOD
    if check_type in CheckType.ALL_NETMON:
        return NETMON_POLLING_PERIOD

    # custom passive check from automation plot
    return parse_period(config.get_value("juggler.checks_period.{}".format(check_type), PASSIVE_CHECK_PERIOD))


def get_check_max_possible_delay(check_type):
    if check_type in CheckType.ALL_NETMON:
        return 2 * get_check_period(check_type)

    elif check_type in CheckType.ALL_HW_WATCHER or check_type in CheckType.OPTIONAL_HW_WATCHER:
        # This is a maximum possible timeout between two consecutive events.
        # HW-watcher's checks produced before `time() - HW_WATCHER_CHECK_AGE_TIMEOUT` should be considered staled.
        return (
            constants.HW_WATCHER_CHECK_PERIOD
            + constants.HW_WATCHER_MAX_PERIOD_DELAY
            + constants.HW_WATCHER_CHECK_ACCURACY
            + constants.PASSIVE_CHECK_PERIOD
            + constants.JUGGLER_PASSTHROUGH_TIME
            + constants.JUGGLER_PUSH_PERIOD
        )

    else:
        # This is a maximum possible timeout between two consecutive events.
        # Passive checks produced before `time() - PASSIVE_CHECK_STALE_TIMEOUT` should be considered staled.
        # This only works for passive checks that have timestamp in their metadata.
        return get_check_period(check_type) + (
            constants.PASSIVE_CHECK_ACCURACY + constants.JUGGLER_PASSTHROUGH_TIME + constants.JUGGLER_PUSH_PERIOD
        )


def get_check_accuracy(check_type):
    if check_type == CheckType.UNREACHABLE:
        return UNREACH_ACCURACY
    if check_type == CheckType.SSH:
        return SSH_CHECK_ACCURACY
    if check_type in CheckType.ALL_HW_WATCHER:
        return HW_WATCHER_CHECK_ACCURACY
    if check_type in CheckType.ALL_PASSIVE:
        return PASSIVE_CHECK_ACCURACY
    if check_type == CheckType.WALLE_RACK:
        return UNREACH_ACCURACY
    if check_type == CheckType.WALLE_RACK_OVERHEAT:
        return HW_WATCHER_CHECK_ACCURACY
    if check_type in CheckType.ALL_NETMON:
        return NETMON_CHECK_ACCURACY

    # custom passive check from automation plot
    return PASSIVE_CHECK_ACCURACY


def _get_checks_to_configure(key):
    checks = config.get_value(key, default=None)

    if not checks:
        return frozenset()

    if not isinstance(checks, (set, frozenset)):
        checks = frozenset(checks)
        config.set_value(key, checks)

    return checks


def get_static_checks_to_configure():
    return _get_checks_to_configure("juggler.extra_checks.default")


def get_project_checks_to_configure(project_id):
    return _get_checks_to_configure("juggler.extra_checks.{}".format(project_id))
