"""Various constants and implementation details.


Juggler implementation details
==============================

Compilation of https://st.yandex-team.ru/SWAT-1130 and personal communication with darkk@ and dldmitry@.


Активные проверки
-----------------

Каждая проверка обладает следующими параметрами:
* Частота проверки R (refresh time).
* Время жизни проверки (TTL = 15 минут).
* Задержка ε (0 < ε < R), которая выбирается случайно.
* Таймаут T, по умолчанию равный 15 секундам. Его можно переопределить при необходимости.

Детали реализации:
* SSH-проверка дергается раз в 1.5 минуты, которая является простейшим TCP-чатом без какой-либо реальной проверки, что
  на удаленном сервере есть все необходимые публичные ключи, туда можно залогиниться, и т. п.
* UNREACHABLE гораздо хитрее:
  * Пинги шлются каждые 5 секунд
  * Таймаут на пинг - 1 секунда
  * Проверка завершается только тогда, когда будет 12 последовательных зафейлившихся или OK пингов. Поэтому на данный
    момент во время флапов проверка может зависнуть на непределенное время (правда через TTL она получит NO DATA) -
    https://st.yandex-team.ru/JUGGLER-371.

После создания проверка запускается в первый раз через ε секунд. Последующие запуски состоятся через i * R секунд, где
i - номер попытки. Если проверка не успевает отработать за T секунд, то ее статус меняется на CRIT.

В случае изменения статуса проверки, результат ее работы сразу же отправляется во frontend. Если состояние проверки
стабильно, то текущее состояние отправляется раз в TTL/2 секунд. Это сделано для уменьшения нагрузки на frontend.

Каждый хост обрабатывается одним и тем же инстансом pongerd в пределах одной ноды. Ноды (jmon01, jmon02, jmon03) при
этом независимы, и на каждой стоит свой собственной pongerd, отправляющий в неё же данные.


Пассивные проверки
------------------

META - это пассивная проверка, которая отправляется самим клиентом. Смысл её в том, что клиент - жив.

Все пассивные проверки срабатывают раз в 5 минут (настраивается). Если не удается отправить данные - повторных попыток
не делается. Правда, не совсем понятно, когда именно эта проверка начнёт отправляться после наливки: проверки прииносит
juggler-client, который сам устанавливается salt-ом. Когда вся эта машинерия прососётся – непонятно.

Для того, чтобы добавить новую пассивную проверку на хост, её надо добавить в бандл. Скоро у wall-e появится свой
собственный бандл, до тех пор наши проверки входят в поисковый коплект, непоиск копирует проверки из svn-а к себе сам.

Все проверки объединены в агрегаты, Wall-E подписан на получение данных по агрегатам. Каждый агрегат объединяет в себя
проверки одного типа по всем хостам из одной стойки. Таким образом, проверки каждого хоста входят в
len(CheckType.ALL_JUGGLER) агрегатов, по одной проверке в каждом агрегате.

Детали реализации:
Агреатор в Juggler-е обрабатывает проверки не при каждом получении состояния, а периодически. Период может задаваться
в настройках агрегата (refresh time), в наших агрегатах 90 сек.

ttl для события тоже задаётся в каждом агрегате, то есть какая-то другая система может настоить себе произвольный ttl
для тех же самых событий, на которые смотрит Wall-E. В агрегатах Wall-E ttl равен 900 сек.

Таким образом, для каждой пассивной проверки задержки такие:
1. Запускается проверка каждые пять минут.
2. Таймаут на выполнение проверки 3 минуты.
3. Задержку, которую вносит процесс отправки события из клиента на сервер, можно не учитывать отдельно,
    сейчас это единицы секунд.
4. Агрегат пересчитывается каждые 90 секунд.
5.
 5.1. Если состояние агрегата не изменилось, то отправка в Wall-E произойдёт через 5 минут
 5.2. Если состояние агрегата изменилось, то отправка должна произойти немедленно.
6. Задержка, которую вносит процесс отправки, измеряется единицами секунд и тоже может не учитываться отдельно.

Таким образом, суммарная разница во времени получения двух последовательных событий может достигать
300 + 180 + 90 + 300 = 870 сек без учёта задержек при отправке и получении,
при условии, что первое событие проскочило весь пайплайн, ни разу не наткнувшись
на задержку, а второе событие собрало максимальные задержки на каждом этапе.
В промежутке между событиями Wall-E будет ериодически получать от Juggler-а первое событие, в худшем случае
Wall-E получит первое событие три раза.

В случае с проверками, основанными на hw-watcher-е, всё ещё интереснее: hw-watcher запускается каждые 10 минут по крону,
но скрипт запуска добавляет случайную задержку продолжительностью от 0 до 9-ти минут (в среднем 4.5 мин).
Таймаут на выполнение проверки hw-watcher-ом тоже равняется трём минутам. Только после этого результат его выполнения
попадает в пассивную проверку, которая запускается каждые 5 минут. Можно предположить, что пассивная проверка
отрабатывает достаточно быстро, потому что по сути ей требуется только прочитать статус из файла.
Таким образом, для проверок на основе hw-watcher-а задержка между последовательными событиями
может достигнуть 600 + 540 + 180 + 300 + 90 + 300 = 2010 сек – без учёта задержек при отправке и получении.


Хосты Wall-E в Juggler
----------------------

В Juggler есть собственные группы хостов, на которые можно мапить CMS'ные, кондукторные и прочие произвольные группы.
Период синхронизации - 20 минут.

Вот этот PR - https://git.qe-infra.yandex-team.ru/projects/JUGGLER/repos/juggler/pull-requests/83 создает две новые
группы WALLE%PROD и WALLE%DEV. В них Juggler добавляет все хосты с продуктивной и тестовой инсталляций Wall-E.


Проверки Wall-E в Juggler
-------------------------

Wall-E периодически пересоздаёт агрегированные проверки в Juggler из модуля walle.juggler для всех хостов,
которые заведены в wall-e. Проверки агрегируются по стойкам, в один агрегат входят проверки одного типа.
Juggler присылает нам изменения состояния проверок пачками в ручку /health/push-statuses,
пачка формируется из 5000 проверок либо из всех имеющихся изменений за 5 секунд
(речь про проверки в теминах Wall-E, это примерно 220 агрегатов по стойкам). При этом, если состояние проверки
изменилось у одного хоста в стойке, то присылается весь агрегат (~40 проверок).


Проверки на основе данных из netmon
-----------------------------------
Wall-E периодически забирает из netmon текущий срез состояния инфраструктуры. Для каждого объекта (свич, очередь, ДЦ)
в этих данных имеется чиселка: средняя связность относительно всех пар, указывающих на данный свитч. Связность
считается так: в каждой ячейке (source_switch, target_switch) считается доля проб с уровнем потерь меньшим заданного,
затем высчитывается взвешенное среднее, где весом выступает количество машин в source_switch.

netmon передаёт связность в виде списка из четырёх значений, каждое значение – это связность с заданным уровнем потерь.
Wall-E смотрит на связность с 10% потерями: если связность меньше, чем 0.9 (переопределяется в конфиге),
то свич считается мёртвым и проверка загорается в FAILED.

Дополнительная логика: проверка так же загорается в FAILED, если связность ДЦ или очереди меньше
того же порога по тем же параметрам. Есть желание прикрутить статус SUSPECTED.


Проверки агрегатов на основе данных о состоянии других проверок
---------------------------------------------------------------
Wall-E делает свою собственную проверку состояния для условного агрегата "серверная стойка в ДЦ". Предполагается, что
данные о физическом расположении серверов, которые Wall-E получает из БОТ-а, позволяют вычислять состояние такого
агрегата в качестве "метапроверки".

Wall-E использует данные проверки UNREACHABLE, сгруппированные по физическому расположению серверов. Если все сервера
в стойке "здоровые" (wall-e смотрит на акутальное состояние проверки для сервера, то есть игнорирует флаподав),
то стойка считается здоровой. Если какой-то процент серверов в стойке умирает (условно, "все сервера"), стойка считается
нездоровой. Есть дополнительное промежуточное состояние: умер определённый процент серверов в стойке,
но не "все сервера", такая стойка переводится в статус SUSPECTED (wall-e перестаёт лечить хосты в такой стойке), но
если после определённого таймаута в стойке всё ещё не умерли "все сервера", то стойка считается здоровой.
Таймаут отсчитывается от максимального status_mtime среди умерших серверов.
status_mtime – это время перехода проверки в текущий статус, в данном случае, проверки UNREACHABLE в статус CRIT.

Как следствие, если в стойке есть сервер, который флапает, Wall-E продолжит считать такую стойку подозрительной и не
будет лечить сервера в ней.


Wall-E implementation details
=============================
Juggler & netmon checks
При получении данных от Juggler, Wall-E по сути всегда оперирует актуальным состоянием хостов – когда он меняется,
Juggler отправляет новое состояние, когда оно не меняется – изменения не отсылаются (с точностью до особенностей
реализации). Для netmon wall-e поддерживает аналогичное поведение на своей стороне: мы считаем, что у нас всегда
имеется текущее состояние сети, а не состояние на какой-то момент времени. Отчасти это связано с тем, как сам netmon
считает состояние сети (оно вычисляется на основе данных за какой-то промежуток времени, а не на основе мгновенного
состояния), отчасти с тем, что поддерживать вечно две разные схемы (длительную для Juggler и мгновенную для netmon)
нет интереса.

Соответственно, проверяя необходимость применения процедур к хосту, Wall-E смотрит на то, в каком состоянии хост
и сеть находятся прямо сейчас. Единственное уточнение: wall-e знает о лагах обеих систем и следит за тем, чтобы время
перехода хоста в текущее состояние билось с этими лагами.

"""


from sepelib.core import constants
from sepelib.core.constants import MINUTE_SECONDS

HEALTH_SCREENING_PERIOD = constants.MINUTE_SECONDS
"""Period with which sync health data is received from HealthDB."""

JUGGLER_AGGREGATES_SYNC_PERIOD = 5 * constants.MINUTE_SECONDS
"""Wall-E runs routine that subscribes Wall-E for host health events."""

_JUGGLER_AGGREGATES_SYNC_RUNTIME = 30 * constants.MINUTE_SECONDS
"""It is a fast routine if there are only few new hosts.
Problems start when there are many new hosts or many new checks. Assume that 30 minutes is enough.
"""

JUGGLER_PUSH_PERIOD = 5 * constants.MINUTE_SECONDS
"""Wall-E configures notifications for host with this period for repeat data."""

NETMON_POLLING_PERIOD = 2 * constants.MINUTE_SECONDS
"""Period for polling netmon."""

HOST_BOOT_TIME = 10 * constants.MINUTE_SECONDS
"""All hosts are monitored with assumption that this time is enough for any OS to fully boot after a hardware reset."""

MAX_DNS_CACHE_TIME = 3 * constants.HOUR_SECONDS
"""Maximum age of a cached DNS record."""


_OVERHEAD = constants.MINUTE_SECONDS
"""Network delay + computation time."""

JUGGLER_PASSTHROUGH_TIME = 5 * constants.MINUTE_SECONDS
"""Juggler client delay (few seconds)
 + client-side or server-side batching (up to 10 seconds)
 + aggregator period (90 sec)
 + banshee delay (few seconds)
 + push sender batch collection time (5 seconds)."""

UNREACH_PERIOD = 5
"""Juggler UNREACHABLE check doesn't have a period,
it has a pongerd built-in flap detection and probes every 5 seconds."""

UNREACH_ACCURACY = constants.MINUTE_SECONDS
"""This is a pongerd built-in flap detection."""

SSH_CHECK_PERIOD = 1.5 * constants.MINUTE_SECONDS
"""Juggler ssh check runs every 1.5 minute with timeout 1 sec."""

SSH_CHECK_ACCURACY = 1
"""Assume check timeout as it's accuracy, 1 sec for ssh check."""

ACTIVE_CHECK_SUSPECTED_PERIOD = 10 * MINUTE_SECONDS
"""Mark checks as suspected instead of failed until this period from check's status_mtime."""

PASSIVE_CHECK_ACCURACY = 3 * constants.MINUTE_SECONDS
"""Maximum delay between check script start moment and actual check data produced.
It is an actual timeout used in juggler-client to terminate checks scripts that run too long.
"""

PASSIVE_CHECK_PERIOD = 5 * constants.MINUTE_SECONDS
"""Juggler passive check (running on Juggler client) period."""

_JUGGLER_CHECK_PERIOD = max(UNREACH_ACCURACY, PASSIVE_CHECK_PERIOD)
"""For simplicity use maximum period."""

# Few words about hw-watcher periodic checks:
# Basically, every 10 minutes cron spawns a script which sleeps for up to 9 minutes and then starts hw-watcher.
# hw-watcher then performs the check which take up to 3 minutes to complete.
# So, the maximum delay between two consecutive checks can be up to 19 minutes which is not a check period:
# check period is 10 minutes (and this should be an average delay as well).

HW_WATCHER_CHECK_PERIOD = 10 * constants.MINUTE_SECONDS
"""hw-watcher checks run by cron every 10 minutes."""

HW_WATCHER_MAX_PERIOD_DELAY = 9 * constants.MINUTE_SECONDS
"""hw-watcher checks run with random delay for up to 9 minutes."""

HW_WATCHER_CHECK_ACCURACY = 3 * constants.MINUTE_SECONDS
"""
hw-watcher check time is a time when the check has been finished. Check running time varies: it may be a few
seconds for link check or up to 3 minutes for disk check when there are too many disks or when check times out on BOT
API."""


CHECK_AGE_TIMEOUT = 2 * JUGGLER_PUSH_PERIOD
"""Checks received before `time() - CHECK_AGE_TIMEOUT` should be considered staled."""

HW_WATCHER_CHECK_MAX_POSSIBLE_DELAY = (
    HW_WATCHER_CHECK_PERIOD
    + HW_WATCHER_MAX_PERIOD_DELAY
    + HW_WATCHER_CHECK_ACCURACY
    + PASSIVE_CHECK_PERIOD
    + JUGGLER_PASSTHROUGH_TIME
    + JUGGLER_PUSH_PERIOD
)
"""This is a maximum possible timeout between two consecutive events.
HW-watcher's checks produced before `time() - HW_WATCHER_CHECK_AGE_TIMEOUT` should be considered staled.
"""

NETMON_CONNECTIVITY_LEVEL_DC = "dc"
NETMON_CONNECTIVITY_LEVEL_QUEUE = "queue"
NETMON_CONNECTIVITY_LEVEL_SWITCH = "switch"

NETMON_CONNECTIVITY_WINDOW_DC = 2 * MINUTE_SECONDS
NETMON_CONNECTIVITY_WINDOW_QUEUE = 9 * MINUTE_SECONDS
NETMON_CONNECTIVITY_WINDOW_SWITCH = 30 * MINUTE_SECONDS
"""Netmon calculates network connectivity status upon data collected inside certain time frame.
Datacenter connectivity uses probes collected in last 2 minutes (there is a lot of them for a datacenter),
while switch connectivity status requires longer observations because there aren't much probes for a switch.
"""

NETMON_CONNECTIVITY_LEVEL = NETMON_CONNECTIVITY_LEVEL_DC
"""We check netmon connectivity for DC, not for switch:
 * switch takes 30 minutes to collect the data, we can not afford it.
 * dc level contains breakdown up to the switch level, it is just less accurate, so we can get all data we need.
 * we have to mark switch as failing as soon as it's connectivity drops to 90% to react faster.
 That means, we'll get some false OKs as well as some false FAILs.
"""
NETMON_CONNECTIVITY_WINDOW = NETMON_CONNECTIVITY_WINDOW_DC

NETMON_CONNECTIVITY_STALE_TIMEOUT = NETMON_POLLING_PERIOD
"""Netmon re-computes this data every 15 seconds, but we can not pull it with such a high frequency.
Still, we can not trust received data if it is older then our polling period.
Treat this case a if it was a netmon outage.
"""

NETMON_CHECK_ACCURACY = NETMON_CONNECTIVITY_WINDOW
"""Accuracy for netmon check, used to calculate effective timestamp."""


HOST_ADOPTION_TIMEOUT = (
    JUGGLER_AGGREGATES_SYNC_PERIOD
    + _JUGGLER_AGGREGATES_SYNC_RUNTIME
    + _JUGGLER_CHECK_PERIOD
    + JUGGLER_PASSTHROUGH_TIME
    + JUGGLER_PUSH_PERIOD
    + HEALTH_SCREENING_PERIOD
    + _OVERHEAD
    # Give a time to handle some temporary errors
    + 30 * constants.MINUTE_SECONDS
)
"""Maximum time that is required for heath data to appear in Wall-E."""


HEALTH_STATUS_FILTER_MISSING = "no"
"""Filter for missing health status."""

HEALTH_STATUS_FILTER_FAILURE = "failure"
"""Filter for failed health status."""

HEALTH_STATUS_FILTER_OK = "ok"
"""Filter for all-green health status."""

HEALTH_STATUS_FILTERS = [HEALTH_STATUS_FILTER_MISSING, HEALTH_STATUS_FILTER_OK, HEALTH_STATUS_FILTER_FAILURE]
"""Subset of possible health statuses for filtering in API."""


MONITORING_TIMEOUT = 2 * _JUGGLER_CHECK_PERIOD + JUGGLER_PASSTHROUGH_TIME + _OVERHEAD + 2 * JUGGLER_PUSH_PERIOD
"""
All hosts are monitored with assumption that this time is enough for Wall-E to obtain a valid host health status.
"""


NETMON_REACTION_TIMEOUT = NETMON_CONNECTIVITY_WINDOW + NETMON_POLLING_PERIOD + _OVERHEAD
"""Assume that netmon needs at most one full time frame of dc level probes to detect a failure."""

RACK_FAILURE_REACTION_TIMEOUT = UNREACH_ACCURACY + JUGGLER_PASSTHROUGH_TIME + _OVERHEAD + JUGGLER_PUSH_PERIOD
"""Assume that Juggler needs at most one active check period to get actual check states for all hosts in a rack.
NB: adding JUGGLER_PUSH_PERIOD as a safety measure. Juggler should push changes to Wall-E as soon as they occur,
but it has too many master switches during the day.
"""

NETWORK_CHECKS_REACTION_TIMEOUT = max(NETMON_REACTION_TIMEOUT, RACK_FAILURE_REACTION_TIMEOUT)
"""This will effectively delay reaction for UNREACHABLE and SSH checks,
Wall-E will wait for this timeout before making a decision."""
