"""Decision class represent DMC's thought result."""

import io

from sepelib.core.exceptions import LogicalError
from walle import restrictions as host_restrictions
from walle.expert.types import WalleAction, Failure, CheckType, get_walle_check_type
from walle.util.misc import drop_none

RESTRICTIONS_MAP = {
    WalleAction.REBOOT: [host_restrictions.AUTOMATED_REBOOT],
    WalleAction.PROFILE: [
        host_restrictions.AUTOMATED_PROFILE,
        host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
    ],
    WalleAction.REDEPLOY: [host_restrictions.AUTOMATED_REDEPLOY],
    WalleAction.REPAIR_HARDWARE: [host_restrictions.AUTOMATED_REPAIRING],
    WalleAction.CHANGE_DISK: [host_restrictions.AUTOMATED_DISK_CHANGE],
    WalleAction.RESET_BMC: [host_restrictions.AUTOMATED_BMC_REPAIR],
    WalleAction.REPAIR_CPU: [
        host_restrictions.AUTOMATED_CPU_REPAIR,
        host_restrictions.AUTOMATED_PROFILE,
        host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
    ],
    WalleAction.REPAIR_MEMORY: [
        host_restrictions.AUTOMATED_MEMORY_REPAIR,
        host_restrictions.AUTOMATED_PROFILE,
        host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
    ],
    WalleAction.REPAIR_REBOOTS: [
        host_restrictions.AUTOMATED_REDEPLOY,
        host_restrictions.AUTOMATED_PROFILE,
        host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
    ],
    WalleAction.REPORT_FAILURE: [host_restrictions.AUTOMATED_HEALING],
    WalleAction.REPAIR_RACK_FAILURE: [host_restrictions.AUTOMATED_RACK_REPAIR],
    WalleAction.REPAIR_RACK_OVERHEAT: [host_restrictions.AUTOMATED_RACK_REPAIR],
    WalleAction.DEACTIVATE: [],
}


class Decision:
    def __init__(
        self,
        action,
        reason,
        params=None,
        checks=None,
        failure_type=None,
        failures=None,
        restrictions=None,
        failure_check_info=None,
        counter=0,
        rule_name="",
    ):
        # TODO(rocco66): remove in favor ORM Decision?
        self.action = action
        self.params = params or None
        self.checks = checks or None
        self.failure_type = failure_type or None

        if failure_type and not isinstance(failure_type, str):
            self.failure_type = failure_type[0]

        self.failures = self._mk_failures(failures, action, checks)
        self.failure_check_info = failure_check_info

        if checks:
            self.walle_checks = [get_walle_check_type(check_type) for check_type in checks]
        else:
            self.walle_checks = None

        self.reason = reason
        self.restrictions = restrictions or []
        self.counter = counter
        self.rule_name = rule_name

    @classmethod
    def healthy(cls, reason, checks=None):
        return cls(WalleAction.HEALTHY, reason, checks=checks)

    @classmethod
    def deactivate(cls, reason, params=None, failure_type=None):
        return cls(WalleAction.DEACTIVATE, reason, params=params, failure_type=failure_type)

    @classmethod
    def wait(cls, reason, checks=None, params=None, failure_type=None):
        return cls(WalleAction.WAIT, reason, checks=checks, params=params, failure_type=failure_type)

    @classmethod
    def failure(cls, reason, params=None, failure_type=None):
        return cls(WalleAction.FAILURE, reason, params=params, failure_type=failure_type)

    def escalate(self, action, reason, params=None, failure_type=None, restrictions=None):
        params = self.params if params is None else params
        failure_type = self.failure_type if failure_type is None else failure_type
        restrictions = self.restrictions if restrictions is None else restrictions

        return Decision(
            action,
            reason,
            params=params,
            checks=self.checks,
            failure_type=failure_type,
            failures=self.failures,
            restrictions=restrictions,
            failure_check_info=self.failure_check_info,
        )

    @classmethod
    def change_disk(cls, reason, params):
        extra_restrictions = []
        # Unknown disks are fixed via host profiling
        if "slot" not in params:
            extra_restrictions.append(host_restrictions.AUTOMATED_PROFILE)

        if params["redeploy"]:
            extra_restrictions.append(host_restrictions.AUTOMATED_REDEPLOY)

        return cls(WalleAction.CHANGE_DISK, reason, params, checks=[CheckType.DISK], restrictions=extra_restrictions)

    def get_restrictions(self):
        try:
            return set(RESTRICTIONS_MAP[self.action] + self.restrictions)
        except KeyError:
            raise LogicalError()

    def get_param(self, param_name, default=None):
        if self.params is not None:
            return self.params.get(param_name, default)

        return default

    def _mk_failures(self, failures, action, checks):
        if failures:
            return failures

        if checks:
            return [get_walle_check_type(check_type) for check_type in checks]

        if action in Failure.ALL_FAILURE_ACTIONS:
            return [action]

        # DMC can produce a decision just to upgrade a task, with no particular failures.
        return None

    def to_dict(self):
        return drop_none(
            {
                "action": self.action,
                "reason": self.reason,
                "params": self.params,
                "checks": self.checks,
                "failures": self.failures,
                "restrictions": self.restrictions,
                "failure_type": self.failure_type,
                "failure_check_info": self.failure_check_info,
                "counter": self.counter,
                "rule_name": self.rule_name,
            }
        )

    def __eq__(self, other):
        if isinstance(other, Decision):
            return (
                (set(self.checks) if self.checks is not None else None)
                == (set(other.checks) if other.checks is not None else None)
                and self.action == other.action
                and self.params == other.params
                and self.restrictions == other.restrictions
                and self.reason == other.reason
            )
        else:
            raise Exception(
                "A Decision object is compared with a non-Decision object which is prohibited for security reasons."
            )

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        stream = io.StringIO()

        stream.write(self.__class__.__name__)
        stream.write("(")

        for attr_id, attr_name in enumerate(
            ("walle_checks", "action", "params", "failure_type", "reason", "restrictions")
        ):
            if attr_id:
                stream.write(", ")

            stream.write(attr_name)
            stream.write("=")
            stream.write(repr(getattr(self, attr_name)))

        stream.write(")")

        return stream.getvalue()
