"""Host failure log."""
from collections import defaultdict
from uuid import uuid4

from mongoengine import StringField, IntField, ListField, BooleanField

from sepelib.mongo.util import register_model
from walle.expert.types import Failure
from walle.models import Document, timestamp
from walle.util.limits import check_limits, get_max_period, CheckResult
from walle.util.misc import drop_none, gevent_idle_iter
from walle.util.mongo import MongoDocument


@register_model
class FailureLog(Document):
    """Logs host failures and actions that have been processed to handle this particular failure.

    Used exclusively for checking automation limits.
    """

    id = StringField(primary_key=True, required=True, help_text="Unique failure ID")
    host_inv = IntField(required=True, help_text="Target host inventory number")
    aggregate = StringField(required=True, help_text="Host aggregate")
    project = StringField(help_text="Related project ID")
    failures = ListField(StringField(), required=True, help_text="A set of failures that has been processed")
    credited_failures = ListField(
        StringField(), help_text="A set of failures that has been processed at expense of a credit"
    )
    action_time = IntField(required=True, help_text="Time when the last action has been taken")
    not_count = BooleanField(help_text="Host is being worked on and should not be counted")

    meta = {
        "collection": "failure_log",
        "indexes": [
            {"name": "host_limits", "fields": ["host_inv", "failures", "action_time"]},
            {"name": "aggregate_limits", "fields": ["aggregate", "failures", "action_time"]},
            {"name": "total_limits", "fields": ["failures", "action_time"]},
        ],
    }


def check_total_action_limits(failure, start_time, limits, project_id=None):
    """Checks total limits got aggregate."""

    if not limits:
        return CheckResult(True)

    group_field = FailureLog.aggregate if failure in Failure.ALL_RACK else FailureLog.host_inv
    time_field_name = FailureLog.action_time.db_field
    group_field_name = group_field.db_field
    project_field_name = FailureLog.project.db_field
    max_period = get_max_period(limits)

    query = drop_none(
        {
            FailureLog.project.db_field: project_id,
            FailureLog.failures.db_field: failure,
            FailureLog.credited_failures.db_field: {"$ne": failure},
            FailureLog.not_count.db_field: {"$exists": False},
            FailureLog.action_time.db_field: {"$gte": max(timestamp() - max_period, start_time)},
        }
    )

    model_collection = MongoDocument.for_model(FailureLog)
    model_entries = model_collection.find(query, (time_field_name, group_field_name, project_field_name))

    result = defaultdict(dict)
    for entry in gevent_idle_iter(model_entries):
        group_id = entry[group_field_name]
        time = entry[time_field_name]
        project = entry[project_field_name]

        cur_max = result[group_id].get("time", float("-inf"))
        result[group_id]["time"] = max(cur_max, time)
        result[group_id]["_id"] = group_id
        result[group_id]["project"] = project

    objs = sorted(result.values(), key=lambda res: res["_id"], reverse=True)

    return check_limits(objs, "time", limits)


def register_failure(host, failures, credited=None, not_count=False):
    failure_id = str(uuid4())
    failure_obj = FailureLog(
        id=failure_id,
        host_inv=host.inv,
        aggregate=host.get_aggregate(),
        project=host.project,
        failures=failures,
        action_time=timestamp(),
    )

    if credited:
        failure_obj.credited_failures = credited
    if not_count:
        failure_obj.not_count = True

    failure_obj.save()
    return failure_id
