"""Rules for host availability checks."""

import logging

from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.common import CheckTypeDescrBase
from .common import (
    AvailabilityCheckRuleBase,
    get_all_results,
    walle_check_names,
    all_failure_reasons,
    get_escalation_rules,
)
from ..types import WalleAction, CheckType

log = logging.getLogger(__name__)


class AvailabilityDescr(CheckTypeDescrBase):
    check_type = CheckType.ALL_AVAILABILITY
    checks_human_name = "Host availability checks"
    healthy_msg = "Host is available."
    suspected_msg = "Host is suspected to be unavailable"


class AvailabilityCheckRule(AvailabilityCheckRuleBase):
    check_type_descr = AvailabilityDescr
    escalation_rules = get_escalation_rules()

    @classmethod
    def make_decision(cls, host, reasons, enabled_checks):
        check_type = cls.check_type_descr.check_type
        availability_results = get_all_results(check_type, host, reasons, enabled_checks)
        if not availability_results:
            msg = "{} aren't enabled for the host.".format(cls.check_type_descr.checks_human_name)
            return Decision.healthy(msg, checks=check_type)

        infrastructure = cls.get_infrastructure(reasons)
        failed_checks, suspected_checks = cls.get_failed_checks(host, infrastructure, availability_results)

        if failed_checks:
            return cls._decision_reboot_host(failed_checks, suspected_checks)

        if suspected_checks:
            return cls._decision_wait_more(suspected_checks, check_type, FailureType.AVAILABILITY)

        return cls._common_decision(availability_results)

    @staticmethod
    def _decision_reboot_host(failed_checks, suspected_checks):
        human_reasons = ", ".join(walle_check_names(failed_checks) + all_failure_reasons(suspected_checks))
        message = "Host is not available: {}.".format(human_reasons)

        checks = list(set(failed_checks) & set(CheckType.ALL_AVAILABILITY))
        return Decision(WalleAction.REBOOT, message, checks=checks, failure_type=FailureType.AVAILABILITY)
