"""Some base classes for rules."""

import logging
import typing as tp

from sepelib.core.exceptions import LogicalError
from walle.expert.decision import Decision
from .utils import get_check_result, is_disabled_check, log_produced_decision
from ..types import WalleAction, CheckStatus, get_walle_check_type

log = logging.getLogger(__name__)


class AbstractRule:
    """This rule represents an interface for DecisionMaker/automation plot instances.
    Each rule does one thing, although it does not mean that it only checks one check.
    It is possible that one check is being used in many rules.
    It is also possible that one rule uses many checks to produce its decision.

    Rule must ensure it's checks are enabled. If checks are not enabled, then decision must be `WalleAction.HEALTHY`.
    """

    def apply(self, host, reasons, enabled_checks: tp.Optional[frozenset[str]]):
        decision = self.make_decision(host, reasons, enabled_checks)
        log_produced_decision(log, host, decision)

        if decision.action in {WalleAction.HEALTHY, WalleAction.WAIT, WalleAction.DEACTIVATE}:
            return decision

        return self.escalate(host, decision)

    def make_decision(self, host, reasons, enabled_checks):
        """Produce decision based on check statuses."""
        raise NotImplementedError

    def escalate(self, host, decision):
        """Escalate decision based on other host and system information."""
        raise NotImplementedError


def apply_and_set_rule_name(rule, *args, **kwargs):
    if isinstance(rule, SingleCheckRule):
        check_rule = rule._check_rule
    elif isinstance(rule, (AbstractRule, CheckRuleInterface)):
        check_rule = rule
    else:
        raise RuntimeError(f"unknown type for rule name extraction {rule=}")

    decision = rule.apply(*args, **kwargs)
    decision.rule_name = type(check_rule).__name__
    return decision


class SingleCheckRule(AbstractRule):
    def __init__(self, check_rule):
        """

        :type check_rule: CheckRuleInterface
        """

        self._check_rule = check_rule
        self._check_type = check_rule.check_type
        super().__init__()

    def apply(self, host, reasons, enabled_checks: tp.Optional[frozenset[str]]):
        check_type = self._check_type
        if is_disabled_check(host, check_type, enabled_checks):
            return Decision.healthy("{} check is not enabled for the host.".format(check_type), checks=[check_type])

        return super().apply(host, reasons, enabled_checks)

    def make_decision(self, host, reasons, enabled_checks):
        check_type = self._check_type

        check_result = get_check_result(reasons, check_type)

        if check_result["status"] != CheckStatus.FAILED:
            return check_common(check_type, check_result)

        return apply_and_set_rule_name(self._check_rule, host, check_result)

    def escalate(self, host, decision):
        return self._check_rule.escalation_rules.escalate(host, decision)

    def __str__(self):
        return "<SingleCheckRule for {} ({})>".format(self._check_rule.__class__.__name__, self._check_type)


class CheckRuleInterface:
    check_type = None
    escalation_rules = None

    def apply(self, host, check_result):
        raise NotImplementedError


def check_common(check_type, result):
    """Makes a common sense decision for statuses."""

    check_name = get_walle_check_type(check_type)

    status = result["status"]
    if status == CheckStatus.PASSED:
        return Decision.healthy("{} check has '{}' status.".format(check_name, status), checks=[check_type])

    elif status == CheckStatus.MISSING:
        return Decision.wait("No data for {} check result.".format(check_name), checks=[check_type])

    elif status == CheckStatus.VOID:
        return Decision.wait("Wall-E has not received {} check.".format(check_name), checks=[check_type])

    elif status == CheckStatus.UNSURE:
        return Decision.wait("{} check is not fresh enough.".format(check_name), checks=[check_type])

    elif status == CheckStatus.INVALID:
        return Decision.wait("{} check has invalid metadata.".format(check_name), checks=[check_type])

    elif status in CheckStatus.ALL_IGNORED:
        return Decision.wait("{} check is {}.".format(check_name, status), checks=[check_type])

    else:
        raise LogicalError()
