"""Rules that can be configured from Automation Plot."""

from walle.expert.checks import get_check_max_possible_delay
from walle.expert.decision import Decision
from walle.models import timestamp
from walle.operations_log.constants import Operation
from walle.util.misc import first
from .base import AbstractRule, check_common
from .escalation import (
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_deactivate,
    escalate_to_redeploy,
    escalate_to_report,
    Predicate,
    task_has_not_helped,
    EscalationRules,
    escalate_to_profile,
)
from .utils import get_check_result, is_disabled_check
from ..types import WalleAction, CheckStatus


def _action_enabled(action_enabled_flag):
    """This function exists for readability and conformity reasons.
    Create Predicate that takes check.action boolean flag and return whether it True or False."""

    return Predicate(lambda decision: bool(action_enabled_flag))


def _escalate_reboot(profile_enabled, redeploy_enabled):
    if profile_enabled:
        return escalate_to_profile
    elif redeploy_enabled:
        return escalate_to_redeploy
    else:
        return escalate_to_report


def _escalate_profile(redeploy_enabled):
    if redeploy_enabled:
        return escalate_to_redeploy
    else:
        return escalate_to_report


def configure_rule(check, prev_check_type):
    check_type = check.name

    if check.wait:

        def produce_decision(reason):
            return Decision(WalleAction.WAIT, reason=reason, checks=[check_type])

    elif check.reboot:

        def produce_decision(reason):
            return Decision(WalleAction.REBOOT, reason=reason, checks=[check_type])

    elif check.profile:

        def produce_decision(reason):
            return Decision(WalleAction.PROFILE, reason=reason, checks=[check_type])

    elif check.redeploy:

        def produce_decision(reason):
            return Decision(WalleAction.REDEPLOY, reason=reason, checks=[check_type])

    elif check.report_failure:

        def produce_decision(reason):
            return Decision(WalleAction.REPORT_FAILURE, reason=reason, checks=[check_type])

    else:

        def produce_decision(reason):
            return Decision(WalleAction.HEALTHY, reason=reason)

    escalation_rules = EscalationRules(
        # reboots escalations
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(
                Operation.POWER_ON.host_status, "Host powering on hasn't helped", "Host failed to power on"
            ),
            action=_escalate_reboot(profile_enabled=check.profile, redeploy_enabled=check.redeploy),
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(Operation.REBOOT.host_status, "Reboot hasn't helped", "Host failed to reboot"),
            action=_escalate_reboot(profile_enabled=check.profile, redeploy_enabled=check.redeploy),
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(
                Operation.PROFILE.host_status, "Host profiling hasn't helped", "Host failed to profile"
            ),
            action=_escalate_profile(redeploy_enabled=check.redeploy),
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(
                Operation.REDEPLOY.host_status, "Redeploying hasn't helped", "Host failed to redeploy"
            ),
            action=escalate_to_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(Operation.REPORT_FAILURE.host_status, "People are not fixing it"),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=limit_reached("max_host_reboots", Operation.REBOOT),
            action=_escalate_reboot(profile_enabled=check.profile, redeploy_enabled=check.redeploy),
        ),
        # profile escalations
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=task_has_not_helped(
                Operation.PROFILE.host_status, "Host profiling hasn't helped", "Host failed to profile"
            ),
            action=_escalate_profile(redeploy_enabled=check.redeploy),
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=task_has_not_helped(
                Operation.REDEPLOY.host_status, "Redeploying hasn't helped", "Host failed to redeploy"
            ),
            action=escalate_to_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=task_has_not_helped(Operation.REPORT_FAILURE.host_status, "People are not fixing it"),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_host_profiles", Operation.PROFILE),
            action=_escalate_profile(redeploy_enabled=check.redeploy),
        ),
        # escalate redeploy to report
        EscalationPoint(
            predicate=action_match(WalleAction.REDEPLOY),
            reason=task_has_not_helped(
                Operation.REDEPLOY.host_status, "Redeploying hasn't helped", "Host failed to redeploy"
            ),
            action=escalate_to_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REDEPLOY),
            reason=task_has_not_helped(Operation.REPORT_FAILURE.host_status, "People are not fixing it"),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REDEPLOY),
            reason=limit_reached("max_host_redeployments", Operation.REDEPLOY),
            action=escalate_to_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPORT_FAILURE),
            reason=task_has_not_helped(Operation.REPORT_FAILURE.host_status, "People are not fixing it"),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPORT_FAILURE),
            reason=limit_reached("max_host_reports", Operation.REPORT_FAILURE),
            action=escalate_to_deactivate,
        ),
    )

    return ConfigurableRule(check_type, produce_decision, escalation_rules, prev_check_type=prev_check_type)


class ConfigurableRule(AbstractRule):
    def __init__(self, check_type, produce_decision, escalation_rules, prev_check_type=None):
        self._check_type = check_type
        self._prev_check_type = prev_check_type

        self._escalation_rules = escalation_rules
        self._produce_decision = produce_decision

        super().__init__()

    def make_decision(self, host, reasons, enabled_checks):
        check_type = self._check_type

        if is_disabled_check(host, check_type, enabled_checks):
            return Decision.healthy("{} check is not enabled for the host.".format(check_type), checks=[check_type])

        check_result = get_check_result(reasons, check_type)
        if check_result["status"] != CheckStatus.FAILED:
            return check_common(check_type, check_result)

        if self._prev_check_type is not None:
            delay_reason = self._delay_reason(check_result, get_check_result(reasons, self._prev_check_type))
            if delay_reason is not None:
                return Decision.wait(reason=delay_reason, checks=[check_type])

        return self._produce_decision(self._failure_reason(check_result))

    def escalate(self, host, decision):
        return self._escalation_rules.escalate(host, decision)

    def _delay_reason(self, check_result, prev_check_result):
        delay_reasons = [
            self._previous_check_needs_to_catch_up(check_result),
            self._previous_check_is_failing(check_result, prev_check_result),
            self._current_check_needs_to_catch_up(check_result, prev_check_result),
        ]

        return first(filter(None, delay_reasons))

    def _failure_reason(self, check_result, message=""):
        return "{} check failed: {}{}.".format(self._check_type, check_result["metadata"]["reason"], message)

    def _previous_check_needs_to_catch_up(self, check_result):
        if _status_mtime_is_too_fresh(self._check_type, check_result["status_mtime"]):
            return self._failure_reason(
                check_result, ", but it may be some other failure. Wait for other checks to catch up"
            )

    def _previous_check_is_failing(self, check_result, prev_check_result):
        if prev_check_result["status"] in {CheckStatus.FAILED, CheckStatus.SUSPECTED}:
            return self._failure_reason(
                check_result, ", but it can be a {} failure. Skipping".format(self._prev_check_type)
            )

    def _current_check_needs_to_catch_up(self, check_result, prev_check_result):
        if _status_mtime_is_too_fresh(self._prev_check_type, prev_check_result.get("status_mtime", 0)):
            return self._failure_reason(
                check_result,
                ", but it may be an echo of {} failure. Wait for check to catch up".format(self._prev_check_type),
            )


def _status_mtime_is_too_fresh(check_type, status_mtime):
    return timestamp() - status_mtime < get_check_max_possible_delay(check_type)
