"""Tools for making decision escalation."""

import logging

from sepelib.core import config
from walle import restrictions as host_restrictions
from walle.admin_requests.constants import RequestTypes
from walle.clients.eine import ProfileMode
from walle.expert.automation import GLOBAL_DNS_AUTOMATION, PROJECT_DNS_AUTOMATION
from walle.expert.failure_types import FailureType
from walle.expert.rules.utils import log_produced_decision, repair_hardware_params
from walle.expert.types import WalleAction
from walle.hosts import TaskType
from walle.operations_log.constants import Operation
from walle.operations_log.operations import check_limits, compare_last_operation
from walle.util.limits import parse_timed_limits

log = logging.getLogger(__name__)


class IEscalationPoint:
    def escalate(self, host, decision):
        """
        Take host and current decision and produce new decision based on escalation rules.

        :type host: Host
        :type decision: Decision

        :rtype: Decision
        """
        raise NotImplementedError


class EscalationRules(IEscalationPoint):
    """Combine a few escalation rules (EscalationPoints) to create a set that can ultimately escalate given decision.
    Each rule can and should have it's own set of escalation rules.
    Each check's escalation rules must ultimately pick one decision.
    """

    def __init__(self, *escalation_points):
        """
        :type escalation_points: IEscalationPoint
        """

        self._escalation_points = escalation_points

    def escalate(self, host, decision):
        for escalation_point in self._escalation_points:
            decision = escalation_point.escalate(host, decision)

        return decision


class EscalationPoint(IEscalationPoint):
    def __init__(self, predicate, reason, action):
        """
        An escalation point with given predicate, escalation reason and action.
        It's almost like given->when->then pattern, where `given` just asserts that conditions are proper.

        :param predicate: if predicate evaluates to True, then escalation is tried, otherwise escalation is skipped.
        :param reason: produce a reason string for escalation or None if escalation should not be executed.
        :param action: function that will be applied to a decision
            should predicate evaluate to True and should there be a reason.

        :type predicate: Predicate
        :type reason: EscalationReason
        :type action: (Decision, str) -> Decision
        """

        self._predicate = predicate
        self._reason = reason
        self._action = action

    def escalate(self, host, decision):
        if self._predicate.evaluate(decision):
            reason = self._reason.check(host, decision)
            if reason is not None:
                decision = self._action(decision, reason)
                log_produced_decision(log, host, decision, log_healthy=True)

        return decision


class Predicate:
    """
    Predicate is a boolean function on Decision which, when given a decision, determines whether
    escalation point is applicable to it.
    """

    def __init__(self, predicate):
        """
        Create Predicate object from a given predicate function.

        :type predicate: (Decision) -> bool
        """
        self.evaluate = predicate

    @staticmethod
    def evaluate(decision):
        """Take a decision and check if this escalation point is applicable to it.

        :type decision: Decision
        :rtype: bool
        """
        raise NotImplementedError

    def __and__(self, other):
        """
        Combine two predicates into a predicate that evaluates to True only when both self and other evaluate to True.

        :type other: Predicate
        :rtype: Predicate

        Example:
        >>> predicate = Predicate(...) & Predicate(...)

        """

        cls = type(self)
        return cls(lambda decision: self.evaluate(decision) and other.evaluate(decision))

    def __invert__(self):
        """Create predicate that evaluates to False when `self` evaluates to True and vice versa.

        :rtype: Predicate

        Example:
        >>> predicate = ~Predicate(...)

        """
        cls = type(self)
        return cls(lambda decision: not self.evaluate(decision))


class EscalationReason:
    """Function that produces a reason for escalation to be executed."""

    def __init__(self, reason_func):
        """
        Wrap given function into an EscalationReason object so that it could be found by class usage.

        :type reason_func: (Host, Decision) -> [str | None]
        """
        self.check = reason_func

    @staticmethod
    def check(host, decision):
        """
        Take host and decision and produce a reason to escalate the decision
        or None if this decision should not be escalated.

        :type host: Host
        :type decision: Decision
        :rtype: str | None
        """
        raise NotImplementedError


def action_match(action):
    """
    Create predicate that checks if decision's action matches with required action.

    :type action: str
    :rtype: Predicate
    """
    return Predicate(lambda decision: decision.action == action)


@EscalationReason
def automatic_profile_not_supported(host, decision):
    """Check that automated profile is enabled for the host's project."""

    project = host.get_project(fields=("id", "profile", "vlan_scheme"))

    if project.profile is None:
        return "{} '{}' project doesn't have a default profile, so automatic profiling can't be used.".format(
            decision.reason, project.id
        )

    if project.vlan_scheme is None:
        return "{} '{}' project doesn't have a VLAN scheme, so automatic profiling can't be used.".format(
            decision.reason, project.id
        )

    return None


def _check_limit(host, operation, limit_name, params=None):
    limits = parse_timed_limits(config.get_value(limit_name))
    return check_limits(host, operation, limits, params=params)


def _limit_operation_params(params, decision):
    if callable(params):
        return params(decision)
    else:
        return params


def limit_reached(limit_name, operation, params=None):
    """
    Create EscalationReason that checks for specified limit on specified operation with given parameters.

    :type limit_name: str|unicode
    :type operation: walle.operations_log.constants.Operation
    :type params: dict | None | (Decision) -> [dict | None]
    :rtype: EscalationReason
    """

    limit_name = "automation.host_limits.{}".format(limit_name)

    @EscalationReason
    def _limit_reached_reason(host, decision):
        check_result = _check_limit(host, operation, limit_name, params=_limit_operation_params(params, decision))
        if not check_result:
            return "{} We tried to {} ({}).".format(decision.reason, operation.operation_name, check_result.info)

        return None

    return _limit_reached_reason


def operation_repeated(operation):
    """
    Create EscalationReason that checks for repeat of specified operation with given parameters.

    :type operation: walle.operations_log.constants.Operation
    :rtype: EscalationReason
    """

    @EscalationReason
    def _operation_repeated_reason(host, decision):
        is_repeated = compare_last_operation(host, operation)
        if is_repeated:
            return "{} We already tried to {} .".format(decision.reason, operation.operation_name)

        return None

    return _operation_repeated_reason


def task_has_not_helped(host_status, reason_for_automated, reason_for_manual=None):
    """
    Create EscalationReason that checks current host status.

    :param host_status: host status that triggers escalation
    :param reason_for_automated: reason string in case task was automated
    :param reason_for_manual: reason string in case task was manual
    :rtype: EscalationReason
    """

    @EscalationReason
    def _task_has_not_helped_reason(host, decision):
        if host.status == host_status:
            if reason_for_manual is None:
                reason = reason_for_automated
            else:
                automated = host.task.type == TaskType.AUTOMATED_HEALING
                reason = reason_for_automated if automated else reason_for_manual

            return "{}: {}".format(reason, decision.reason)

        return None

    return _task_has_not_helped_reason


def platform_not_supported(operation):
    """
    Create EscalationReason that checks host platform support target operation.

    :type operation: walle.operations_log.constants.Operation
    :rtype: EscalationReason
    """

    config_name = "automation.platform_support.{}".format(operation.type)

    @EscalationReason
    def _platform_not_supported_reason(host, decision):
        settings = config.get_value(config_name)

        host_system = host.platform.system if host.platform else None
        host_board = host.platform.board if host.platform else None

        exclude = settings.get("exclude")
        platforms = settings.get("platforms")
        match = False

        for platform in platforms:
            if ("system" not in platform or host_system == platform["system"]) and (
                "board" not in platform or host_board == platform["board"]
            ):
                match = True
                break

        if match and not exclude or not match and exclude:
            return None

        return "{} Platform not supported {}".format(decision.reason, operation.operation_name)

    return _platform_not_supported_reason


@EscalationReason
def dns_automation_off(host, decision):
    if not GLOBAL_DNS_AUTOMATION.is_enabled():
        return "{} Global DNS automation is off".format(decision.reason)

    if not PROJECT_DNS_AUTOMATION.enabled_for_project_id(host.project):
        return "{} Project DNS automation is off".format(decision.reason)

    try:
        host_restrictions.check_restrictions(host, host_restrictions.AUTOMATED_DNS)
    except host_restrictions.OperationRestrictedError:
        return "{} Host has restriction for DNS automation".format(decision.reason)

    return None


def escalate_to_profile(decision, reason):
    return decision.escalate(WalleAction.PROFILE, reason)


def escalate_to_highload_test(decision, reason):
    return decision.escalate(WalleAction.PROFILE, reason, params={"profile_mode": ProfileMode.HIGHLOAD_TEST})


def escalate_to_extra_highload_test(decision, reason):
    return decision.escalate(WalleAction.PROFILE, reason, params={"profile_mode": ProfileMode.EXTRA_HIGHLOAD_TEST})


def escalate_to_dangerous_highload_test(decision, reason):
    return decision.escalate(
        WalleAction.PROFILE,
        reason,
        params={"profile_mode": ProfileMode.DANGEROUS_HIGHLOAD_TEST},
        restrictions=[host_restrictions.AUTOMATED_PROFILE, host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP],
    )


def escalate_to_second_time_node_report(decision, reason):
    new_reason = "Host passes profile successfully but the problem does not go away. {}".format(reason)
    return decision.escalate(
        WalleAction.REPAIR_HARDWARE,
        new_reason,
        failure_type=FailureType.SECOND_TIME_NODE,
        params=repair_hardware_params(
            operation_type=Operation.REPORT_SECOND_TIME_NODE.type,
            request_type=RequestTypes.SECOND_TIME_NODE.type,
            redeploy=True,
            power_on_before_repair=True,
        ),
    )


def escalate_to_redeploy(decision, reason):
    return decision.escalate(WalleAction.REDEPLOY, reason)


def escalate_to_report(decision, reason):
    return decision.escalate(WalleAction.REPORT_FAILURE, reason)


def escalate_to_deactivate(decision, reason):
    return decision.deactivate(reason)


def escalate_to_repair_overheat(decision, reason):
    params = repair_hardware_params(
        request_type=RequestTypes.CPU_OVERHEATED.type,
        operation_type=Operation.REPAIR_OVERHEAT.type,
        reboot=True,
    )
    return decision.escalate(
        WalleAction.REPAIR_HARDWARE,
        reason,
        params=params,
        failure_type=FailureType.CPU_OVERHEATED,
        restrictions=[host_restrictions.AUTOMATED_OVERHEAT_REPAIR],
    )
