"""Rules for hw_watcher bmc check."""

import logging

from walle import restrictions
from walle.admin_requests.constants import RequestTypes
from walle.clients.eine import ProfileMode
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.common import _is_extra_highload
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_deactivate,
    task_has_not_helped,
    escalate_to_extra_highload_test,
    escalate_to_second_time_node_report,
)
from walle.expert.rules.hw_watcher_rules.util import get_eine_code
from walle.expert.rules.utils import repair_hardware_params
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation
from walle.util.misc import drop_none

log = logging.getLogger(__name__)


REASON_FAILURE_MAPPING = {
    "ipmi": FailureType.BMC_IPMI,
    "battery": FailureType.BMC_BATTERY,
    "bmc": FailureType.BMC_IP_DNS,
    "12V": FailureType.BMC_VOLTAGE,
}


def _get_bmc_failure_hw_watcher_result(check_result):
    hw_watcher_result = check_result["metadata"]["result"]
    failure_type = _get_bmc_failure_type(hw_watcher_result)
    return failure_type, hw_watcher_result


def escalate_to_repair_bmc_ipmi(decision, reason):
    params = repair_hardware_params(
        operation_type=Operation.REPAIR_BMC.type, request_type=RequestTypes.IPMI_UNREACHABLE.type
    )

    return decision.escalate(
        WalleAction.REPAIR_HARDWARE,
        reason,
        params=params,
        failure_type=FailureType.BMC_IPMI,
        restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
    )


def _get_bmc_failure_type(hw_watcher_result):
    if len(hw_watcher_result["reason"]) == 1:
        reason_string = hw_watcher_result["reason"][0]

        if reason_string.startswith(("ipmi: ", "battery: ", "bmc: ", "12V: ")):
            reason_failure_type = reason_string.split(":")[0]
            return REASON_FAILURE_MAPPING.get(reason_failure_type, FailureType.BMC_OTHER)

    return FailureType.BMC_OTHER


class CheckBmcIpmi(CheckRuleInterface):

    check_type = CheckType.BMC

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.RESET_BMC),
            reason=task_has_not_helped(Operation.RESET_BMC.host_status, "BMC reset cold didn't help"),
            action=escalate_to_repair_bmc_ipmi,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.RESET_BMC),
            reason=limit_reached("max_reset_bmcs", Operation.RESET_BMC),
            action=escalate_to_repair_bmc_ipmi,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_bmcs", Operation.REPAIR_BMC),
            action=escalate_to_extra_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE) & _is_extra_highload,
            reason=limit_reached(
                "max_host_extra_highload_profiles", Operation.PROFILE, params={"modes": ProfileMode.EXTRA_HIGHLOAD_TEST}
            ),
            action=escalate_to_second_time_node_report,
        ),
    )

    def apply(self, host, check_result):
        failure, hw_watcher_result = _get_bmc_failure_hw_watcher_result(check_result)
        if failure == FailureType.BMC_IPMI:
            return self._decision_reset_bmc(hw_watcher_result, check_result)
        else:
            return Decision.healthy("Host is healthy.")

    @staticmethod
    def _decision_reset_bmc(hw_watcher_result, check_result):
        reason = "hw-watcher: {reason}".format(reason=" ".join(hw_watcher_result["reason"]))

        params = drop_none(
            {
                "request_type": RequestTypes.IPMI_UNREACHABLE.type,
                "eine_code": get_eine_code(hw_watcher_result),
            }
        )
        return Decision(
            WalleAction.RESET_BMC,
            reason,
            params,
            checks=[CheckType.BMC],
            failure_type=FailureType.BMC_IPMI,
            failure_check_info=check_result,
        )


class CheckBmcIpDns(CheckRuleInterface):

    check_type = CheckType.BMC

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_bmcs", Operation.REPAIR_BMC),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        failure, hw_watcher_result = _get_bmc_failure_hw_watcher_result(check_result)
        if failure == FailureType.BMC_IP_DNS:
            return self._decision_bmc_ip_dns_failed(hw_watcher_result, check_result)
        else:
            return Decision.healthy("Host is healthy.")

    @staticmethod
    def _decision_bmc_ip_dns_failed(hw_watcher_result, check_result):
        reason = "hw-watcher: {reason}".format(reason=" ".join(hw_watcher_result["reason"]))

        params = repair_hardware_params(
            request_type=RequestTypes.IPMI_HOST_MISSING.type,
            eine_code=get_eine_code(hw_watcher_result),
            operation_type=Operation.REPAIR_BMC.type,
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params,
            checks=[CheckType.BMC],
            failure_type=FailureType.BMC_IP_DNS,
            restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
            failure_check_info=check_result,
        )


class CheckBmcBattery(CheckRuleInterface):

    check_type = CheckType.BMC

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_bmcs", Operation.REPAIR_BMC),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        failure, hw_watcher_result = _get_bmc_failure_hw_watcher_result(check_result)
        if failure == FailureType.BMC_BATTERY:
            return self._decision_bmc_battery_failed(hw_watcher_result, check_result)
        else:
            return Decision.healthy("Host is healthy.")

    @staticmethod
    def _decision_bmc_battery_failed(hw_watcher_result, check_result):
        reason = "hw-watcher: {reason}".format(reason=" ".join(hw_watcher_result["reason"]))

        params = repair_hardware_params(
            request_type=RequestTypes.BMC_LOW_BATTERY.type,
            eine_code=get_eine_code(hw_watcher_result),
            operation_type=Operation.REPAIR_BMC.type,
            reboot=True,
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params,
            checks=[CheckType.BMC],
            failure_type=FailureType.BMC_BATTERY,
            restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
            failure_check_info=check_result,
        )


class CheckBmcVoltage(CheckRuleInterface):

    check_type = CheckType.BMC

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_bmcs", Operation.REPAIR_BMC),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        failure, hw_watcher_result = _get_bmc_failure_hw_watcher_result(check_result)
        if failure == FailureType.BMC_VOLTAGE:
            return self._decision_bmc_voltage_failed(hw_watcher_result, check_result)
        else:
            return Decision.healthy("Host is healthy.")

    @staticmethod
    def _decision_bmc_voltage_failed(hw_watcher_result, check_result):
        reason = "hw-watcher: {reason}".format(reason=" ".join(hw_watcher_result["reason"]))

        params = repair_hardware_params(
            request_type=RequestTypes.BMC_LOW_VOLTAGE.type,
            eine_code=get_eine_code(hw_watcher_result),
            operation_type=Operation.REPAIR_BMC.type,
            reboot=True,
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params,
            checks=[CheckType.BMC],
            failure_type=FailureType.BMC_VOLTAGE,
            restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
            failure_check_info=check_result,
        )


class CheckBmcUnknown(CheckRuleInterface):

    check_type = CheckType.BMC

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=task_has_not_helped(
                Operation.PROFILE.host_status, "Profiling hasn't helped", "Host profiling failed"
            ),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_host_profiles", Operation.PROFILE),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        failure, hw_watcher_result = _get_bmc_failure_hw_watcher_result(check_result)
        if failure == FailureType.BMC_OTHER:
            return self._decision_profile(hw_watcher_result, check_result)
        else:
            return Decision.healthy("Host is healthy.")

    @staticmethod
    def _decision_profile(hw_watcher_result, check_result):
        reason = "hw-watcher: {reason}".format(reason=" ".join(hw_watcher_result["reason"]))

        return Decision(
            WalleAction.PROFILE,
            reason,
            failure_type=FailureType.BMC_OTHER,
            checks=[CheckType.BMC],
            failure_check_info=check_result,
        )
