"""Rules for hw_watcher cpu_caches check."""

import logging

from walle import restrictions
from walle.admin_requests.constants import RequestTypes
from walle.clients.eine import ProfileMode
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.common import _is_extra_highload
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    task_has_not_helped,
    escalate_to_second_time_node_report,
    escalate_to_extra_highload_test,
    escalate_to_report,
)
from walle.expert.rules.hw_watcher_rules.util import get_eine_code, get_reason_from_hw_watcher
from walle.expert.rules.utils import repair_hardware_params
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation
from walle.util.misc import filter_dict_keys

log = logging.getLogger(__name__)


class CheckCpuCaches(CheckRuleInterface):
    check_type = CheckType.CPU_CACHES

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=task_has_not_helped(Operation.REPAIR_CPU.host_status, "CPU change hasn't helped"),
            action=escalate_to_extra_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached(
                "max_repaired_cpu",
                Operation.REPAIR_CPU,
                params=lambda decision: filter_dict_keys(decision.params or {}, {"slot"}) or None,
            ),
            action=escalate_to_extra_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE) & _is_extra_highload,
            reason=limit_reached(
                "max_host_extra_highload_profiles", Operation.PROFILE, params={"modes": ProfileMode.EXTRA_HIGHLOAD_TEST}
            ),
            action=escalate_to_second_time_node_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE) & ~_is_extra_highload,
            reason=limit_reached("max_host_profiles", Operation.PROFILE),
            action=escalate_to_report,
        ),
    )

    def apply(self, host, check_result):
        hw_watcher_result = check_result["metadata"]["result"]
        if get_eine_code(hw_watcher_result) == ["CPU_OVERHEATING"]:
            return Decision.healthy("Host is healthy.")

        reason = get_reason_from_hw_watcher("cpu", hw_watcher_result["reason"])

        eine_code = get_eine_code(hw_watcher_result)
        if eine_code and "SYSTEM_MISCONFIGURED" in eine_code:
            return self._mk_profile_decision(reason, check_result)

        return self._mk_decision(hw_watcher_result, reason, check_result)

    def _mk_profile_decision(self, reason, check_result):
        params = {"profile_mode": ProfileMode.HIGHLOAD_TEST}
        return Decision(
            WalleAction.PROFILE,
            reason=reason,
            params=params,
            failure_type=FailureType.CPU_SYSTEM_MISCONFIGURED,
            checks=[CheckType.CPU_CACHES],
            restrictions=[restrictions.AUTOMATED_CPU_REPAIR],
            failure_check_info=check_result,
        )

    def _mk_decision(self, hw_watcher_result, reason, check_result):
        params = repair_hardware_params(
            operation_type=Operation.REPAIR_CPU.type,
            request_type=RequestTypes.CPU_FAILED.type,
            eine_code=get_eine_code(hw_watcher_result),
            redeploy=True,
            errors=self._errors(hw_watcher_result),
            slot=hw_watcher_result["socket"],
        )

        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason=reason,
            params=params,
            checks=[self.check_type],
            failure_type=FailureType.CPU_FAILURE,
            restrictions=[restrictions.AUTOMATED_CPU_REPAIR, restrictions.AUTOMATED_REDEPLOY],
            failure_check_info=check_result,
        )

    @staticmethod
    def _errors(hw_watcher_result):
        errors = hw_watcher_result["reason"]

        if "raw" in hw_watcher_result:
            errors.append(hw_watcher_result["raw"])

        return errors
