"""Rules for hw_watcher cpu_thermal check."""

import logging

from walle.clients.eine import ProfileMode
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.common import _is_extra_highload
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_extra_highload_test,
    task_has_not_helped,
    escalate_to_repair_overheat,
    escalate_to_second_time_node_report,
)
from walle.expert.rules.hw_watcher_rules.util import get_eine_code, get_reason_from_hw_watcher
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation

log = logging.getLogger(__name__)


class CheckCpuThermalHWW(CheckRuleInterface):
    check_type = CheckType.CPU_CACHES

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=task_has_not_helped(Operation.PROFILE.host_status, "Profile didn't help"),
            action=escalate_to_repair_overheat,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_overheat_profiles", Operation.PROFILE, {"modes": ProfileMode.HIGHLOAD_TEST}),
            action=escalate_to_repair_overheat,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_overheats", Operation.REPAIR_OVERHEAT),
            action=escalate_to_extra_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE) & _is_extra_highload,
            reason=limit_reached(
                "max_host_extra_highload_profiles", Operation.PROFILE, params={"modes": ProfileMode.EXTRA_HIGHLOAD_TEST}
            ),
            action=escalate_to_second_time_node_report,
        ),
    )

    def apply(self, host, check_result):
        try:
            hw_watcher_result = check_result["metadata"]["result"]
            if get_eine_code(hw_watcher_result) != ["CPU_OVERHEATING"]:
                return Decision.healthy("Host is healthy.")

            reason = get_reason_from_hw_watcher("cpu", hw_watcher_result["reason"])
            params = {"profile_mode": ProfileMode.HIGHLOAD_TEST}
            return Decision(
                WalleAction.PROFILE,
                reason,
                params=params,
                failure_type=FailureType.CPU_OVERHEATED,
                checks=[CheckType.CPU_CACHES],
                failure_check_info=check_result,
            )
        except KeyError as e:
            # should it even be here?
            log.error("CheckCpuThermalHWW key error: %s", e)
            return Decision.healthy("Host is healthy.")
