"""Rules for hw_watcher infiniband check."""
from walle import restrictions
from walle.admin_requests.constants import RequestTypes
from walle.clients.eine import ProfileMode
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_report,
)
from walle.expert.rules.hw_watcher_rules.util import (
    get_eine_code,
    get_reason_from_hw_watcher,
    operation_match,
    get_hw_watcher_result,
)
from walle.expert.rules.utils import repair_hardware_params
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation


class INFINIBAND_EINE_CODE:
    INFINIBAND_MISMATCH = "INFINIBAND_MISMATCH"
    INFINIBAND_INVALID_PHYS_STATE = "INFINIBAND_INVALID_PHYS_STATE"
    PCIE_DEVICE_BANDWIDTH_TOO_LOW = "PCIE_DEVICE_BANDWIDTH_TOO_LOW"
    INFINIBAND_ERR = "INFINIBAND_ERR"
    INFINIBAND_MISCONFIGURED = "INFINIBAND_MISCONFIGURED"
    INFINIBAND_INVALID_STATE = "INFINIBAND_INVALID_STATE"
    INFINIBAND_LOW_SPEED = "INFINIBAND_LOW_SPEED"


def _get_failure_type_request_type(hw_watcher_result):
    eine_code = get_eine_code(hw_watcher_result)

    if not eine_code:
        return None, None

    if INFINIBAND_EINE_CODE.INFINIBAND_MISMATCH in eine_code:
        return FailureType.INFINIBAND_MISMATCH, RequestTypes.INFINIBAND_MISMATCH.type
    if INFINIBAND_EINE_CODE.INFINIBAND_INVALID_PHYS_STATE in eine_code:
        return FailureType.INFINIBAND_INVALID_PHYS_STATE, RequestTypes.INFINIBAND_INVALID_PHYS_STATE.type
    if INFINIBAND_EINE_CODE.PCIE_DEVICE_BANDWIDTH_TOO_LOW in eine_code:
        return FailureType.PCIE_DEVICE_BANDWIDTH_TOO_LOW, RequestTypes.PCIE_DEVICE_BANDWIDTH_TOO_LOW.type
    if INFINIBAND_EINE_CODE.INFINIBAND_ERR in eine_code:
        return FailureType.INFINIBAND_ERR, RequestTypes.INFINIBAND_ERR.type
    if INFINIBAND_EINE_CODE.INFINIBAND_MISCONFIGURED in eine_code:
        return FailureType.INFINIBAND_MISCONFIGURED, None
    if INFINIBAND_EINE_CODE.INFINIBAND_INVALID_STATE in eine_code:
        return FailureType.INFINIBAND_INVALID_STATE, RequestTypes.INFINIBAND_INVALID_STATE.type
    if INFINIBAND_EINE_CODE.INFINIBAND_LOW_SPEED in eine_code:
        return FailureType.INFINIBAND_LOW_SPEED, RequestTypes.INFINIBAND_LOW_SPEED.type

    return None, None


class CheckInfiniband(CheckRuleInterface):
    check_type = CheckType.INFINIBAND

    escalation_rules = EscalationRules(
        # we don't know right limits yet
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE) & operation_match(Operation.REPAIR_INFINIBAND),
            reason=limit_reached("max_host_infiniband_failures", Operation.REPAIR_INFINIBAND),
            action=escalate_to_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_host_profiles", Operation.PROFILE),
            action=escalate_to_report,
        ),
    )

    def apply(self, host, check_result):
        # TODO: temporary crutch from WALLE-3905

        hw_watcher_result = get_hw_watcher_result(check_result)

        reason = get_reason_from_hw_watcher("infiniband", hw_watcher_result["reason"])
        failure_type, request_type = _get_failure_type_request_type(hw_watcher_result)

        if request_type is not None:
            return self._mk_repair_decision(failure_type, request_type, hw_watcher_result, reason, check_result)
        return self._mk_profile_decision(failure_type, hw_watcher_result, reason, check_result)

    def _mk_profile_decision(self, failure_type, hw_watcher_result, reason, failure_check_info):
        params = repair_hardware_params(
            request_type=None,
            operation_type=Operation.PROFILE.type,
            eine_code=get_eine_code(hw_watcher_result),
            errors=hw_watcher_result["reason"],
            profile_mode=ProfileMode.FIRMWARE_UPDATE,
        )
        return Decision(
            WalleAction.PROFILE,
            reason=reason,
            params=params,
            checks=[self.check_type],
            failure_type=failure_type,
            restrictions=[restrictions.AUTOMATED_INFINIBAND_REPAIR, restrictions.AUTOMATED_PROFILE],
            failure_check_info=failure_check_info,
        )

    def _mk_repair_decision(self, failure_type, request_type, hw_watcher_result, reason, failure_check_info):
        params = repair_hardware_params(
            request_type=request_type,
            operation_type=Operation.REPAIR_INFINIBAND.type,
            eine_code=get_eine_code(hw_watcher_result),
            errors=hw_watcher_result["reason"],
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason=reason,
            params=params,
            checks=[self.check_type],
            failure_type=failure_type,
            restrictions=[restrictions.AUTOMATED_INFINIBAND_REPAIR],
            failure_check_info=failure_check_info,
        )
