"""Rules for hw_watcher link check."""

import logging

from walle import restrictions
from walle.admin_requests.constants import RequestTypes
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_deactivate,
)
from walle.expert.rules.hw_watcher_rules.util import get_eine_code, get_reason_from_hw_watcher
from walle.expert.rules.utils import repair_hardware_params
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation

log = logging.getLogger(__name__)


class LINK_EINE_CODE:
    PCIE_DEVICE_BANDWIDTH_TOO_LOW = "PCIE_DEVICE_BANDWIDTH_TOO_LOW"
    NIC_ERR = "NIC_ERR"


class CheckLink(CheckRuleInterface):
    check_type = CheckType.LINK

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_links", Operation.REPAIR_LINK),
            action=escalate_to_deactivate,
        )
    )

    def apply(self, host, check_result):
        hw_watcher_result = check_result["metadata"]["result"]
        eine_code = get_eine_code(hw_watcher_result) or []
        reason = get_reason_from_hw_watcher("Link", hw_watcher_result["reason"])

        if LINK_EINE_CODE.PCIE_DEVICE_BANDWIDTH_TOO_LOW in eine_code or LINK_EINE_CODE.NIC_ERR in eine_code:
            return self._decision_link_rx_crc_errors(reason, eine_code, check_result)

        return self._decision_link_default(reason, eine_code, check_result)

    @staticmethod
    def _decision_link_rx_crc_errors(reason, eine_code, failure_check_info):
        params = repair_hardware_params(
            request_type=RequestTypes.MALFUNCTIONING_LINK_RX_CRC_ERRORS.type,
            operation_type=Operation.REPAIR_LINK.type,
            eine_code=eine_code,
            reboot=True,
        )

        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params=params,
            checks=[CheckType.LINK],
            failure_type=FailureType.LINK_RX_CRC_ERRORS,
            restrictions=[restrictions.AUTOMATED_LINK_REPAIR, restrictions.AUTOMATED_REBOOT],
            failure_check_info=failure_check_info,
        )

    @staticmethod
    def _decision_link_default(reason, eine_code, failure_check_info):
        params = repair_hardware_params(
            request_type=RequestTypes.MALFUNCTIONING_LINK.type,
            operation_type=Operation.REPAIR_LINK.type,
            eine_code=eine_code,
        )

        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params=params,
            checks=[CheckType.LINK],
            failure_type=FailureType.LINK_MALFUNCTION,
            restrictions=[restrictions.AUTOMATED_LINK_REPAIR],
            failure_check_info=failure_check_info,
        )
