"""Rule for hosts with broken Juggler agent
It works when host is available (active Juggler checks work), but passive checks are missing
"""

import logging

from sepelib.core.constants import HOUR_SECONDS
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.availability_check import AvailabilityCheckRule, AvailabilityDescr
from walle.expert.rules.common import (
    CheckTypeDescrBase,
    AvailabilityCheckRuleBase,
    get_all_results,
    get_escalation_rules,
)
from walle.expert.rules.escalation import escalate_to_deactivate
from walle.expert.types import WalleAction, Failure, CheckType, CheckStatus, CheckSets
from walle.models import timestamp

log = logging.getLogger(__name__)

MISSING_CHECKS_MIN_AGE = 10 * HOUR_SECONDS


class MissingPassiveChecksDescr(CheckTypeDescrBase):
    check_type = (set(CheckType.ALL_PASSIVE) & CheckSets.FULL_FEATURED) - (
        {CheckType.TOR_LINK} | set(CheckType.ALL_IB)
    )  # exclude META check
    checks_human_name = "Passive"
    healthy_msg = "Host is healthy."
    suspected_msg = "Host is suspected to have missing passive checks"


class MissingPassiveChecksRule(AvailabilityCheckRuleBase):
    check_type_descr = MissingPassiveChecksDescr
    escalation_rules = get_escalation_rules(final_escalation=escalate_to_deactivate)

    @classmethod
    def make_decision(cls, host, reasons, enabled_checks):
        enabled_checks = frozenset(set(enabled_checks))
        # check if both needed check types are enabled
        for check_type_descr in (AvailabilityDescr, MissingPassiveChecksDescr):
            disabled_checks = cls._get_disabled_checks(check_type_descr.check_type, enabled_checks)
            if disabled_checks:
                return Decision.healthy(cls._get_disabled_checks_msg(disabled_checks))

        # run AvailabilityCheck to check that host is available
        avail_decision = cls._get_availability_check_results(host, reasons, enabled_checks)
        if avail_decision.action != WalleAction.HEALTHY:
            return avail_decision

        # check passive checks results
        check_type = cls.check_type_descr.check_type
        passive_results = get_all_results(check_type, host, reasons, enabled_checks)
        _, suspected_checks = cls.get_failed_checks(host, cls.get_infrastructure(reasons), passive_results)
        if suspected_checks:
            return cls._decision_wait_more(suspected_checks, check_type, FailureType.MISSING_PASSIVE_CHECKS)

        if cls._all_checks_are_missing(passive_results):
            if cls._all_checks_are_old_enough(passive_results):
                return cls._decision_reboot_host()
            else:
                return Decision.wait(
                    "Host is suspected to miss all passive checks", failure_type=FailureType.MISSING_PASSIVE_CHECKS
                )

        return Decision.healthy("Host is healthy")

    @classmethod
    def _decision_reboot_host(cls):
        return Decision(
            WalleAction.REBOOT,
            "All passive checks are missing",
            failures=[Failure.CHECKS_MISSING],
            checks=sorted(cls.check_type_descr.check_type),
            failure_type=FailureType.MISSING_PASSIVE_CHECKS,
        )

    @staticmethod
    def _get_disabled_checks(check_type, enabled_checks):
        return set(check_type) - set(enabled_checks)

    @staticmethod
    def _get_disabled_checks_msg(disabled_checks):
        return "Checks {} aren't enabled for the host.".format(", ".join(sorted(disabled_checks)))

    @staticmethod
    def _get_availability_check_results(host, reasons, enabled_checks):
        return AvailabilityCheckRule.make_decision(host, reasons, enabled_checks)

    @staticmethod
    def _all_checks_are_missing(results):
        return all(details["status"] == CheckStatus.MISSING for _, details in results)

    @staticmethod
    def _all_checks_are_old_enough(results):
        old_enough_ts = timestamp() - MISSING_CHECKS_MIN_AGE
        return all(details["status_mtime"] <= old_enough_ts for _, details in results)
