"""A few utils for using in rules."""

import typing as tp

import six
import xxhash

from sepelib.core import config
from walle.application import app
from walle.expert.types import WalleAction, CheckType, CheckStatus
from walle.util.cache import cached_with_exceptions
from walle.util.log_tools import wrap_logging_helper
from walle.util.misc import drop_none

CHECK_OVERRIDES_VALUE_TIMEOUT = 30
CHECK_OVERRIDES_ERROR_TIMEOUT = 15


def repair_hardware_params(
    request_type, operation_type, eine_code=None, reboot=None, redeploy=None, power_on_before_repair=None, **kwargs
):
    """Convenient function for keeping track of common decision parameters for repair hardware task."""

    return drop_none(
        dict(
            {
                "request_type": request_type,
                "operation": operation_type,
                "eine_code": eine_code,
                "reboot": reboot,
                "redeploy": redeploy,
                "power_on_before_repair": power_on_before_repair,
            },
            **kwargs
        )
    )


def flatten_reason(reason):
    if isinstance(reason, str):
        return reason
    else:
        return ", ".join(reason)


def get_check_result(reasons, check_type):
    try:
        return reasons[check_type]
    except KeyError:
        return {"status": CheckStatus.VOID}


def is_disabled_check(host, check_type, enabled_checks: tp.Optional[frozenset[str]]):
    return check_type not in enabled_checks or _should_be_disabled_check(host, check_type)


def _should_be_disabled_check(host, check_type):
    """
    This is a special place for hotfixes and experiments where we can force Wall-E to consider a check to be disabled
    for individual hosts.
    """
    # https://st.yandex-team.ru/HOSTMAN-226
    if host.project == "rtc-testing" and check_type == CheckType.REBOOTS:
        return True

    if check_type in {CheckType.CPU_CACHES} and host.platform and host.platform.system == "R110-B10":
        # keep until https://st.yandex-team.ru/HWWATCHER-152
        return True

    # check if check_type was disabled for this specific host
    hosts_w_disabled_checks = config.get_value("automation.hosts_with_disabled_checks")
    disabled_checks = hosts_w_disabled_checks.get(host.inv, [])
    if check_type in disabled_checks:
        return True

    checks_percentage_overrides = _get_checks_percentage_overrides()

    host_hash = _get_host_hash(host)

    if check_type in checks_percentage_overrides:
        if host_hash >= checks_percentage_overrides[check_type]:
            return True
    else:
        if host_hash >= config.get_value("automation.checks_percentage.{}".format(check_type), 100):
            return True

    return False


def _get_host_hash(host):
    return xxhash.xxh64_intdigest(six.ensure_binary(host.name) if host.name is not None else b'') % 100


@cached_with_exceptions(value_ttl=CHECK_OVERRIDES_VALUE_TIMEOUT, error_ttl=CHECK_OVERRIDES_ERROR_TIMEOUT)
def _get_checks_percentage_overrides():
    return app.settings().checks_percentage_overrides


@wrap_logging_helper
def log_produced_decision(logger, host, decision, log_healthy=False):
    if decision.action != WalleAction.HEALTHY or log_healthy:
        logger.debug("%s: Decision has been made for the host: %s", host.human_id(), decision)
