"""Rules for some other checks in Wall-E's bundle."""

import logging

from walle import restrictions as host_restrictions
from walle.admin_requests.constants import RequestTypes
from walle.clients.eine import ProfileMode
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.utils import flatten_reason, repair_hardware_params
from walle.operations_log.constants import Operation
from .base import CheckRuleInterface
from .escalation import (
    EscalationRules,
    escalate_to_deactivate,
    EscalationPoint,
    action_match,
    limit_reached,
    task_has_not_helped,
    automatic_profile_not_supported,
    escalate_to_highload_test,
    escalate_to_second_time_node_report,
)
from ..types import WalleAction, CheckType

log = logging.getLogger(__name__)


class CheckTaintedKernel(CheckRuleInterface):
    check_type = CheckType.TAINTED_KERNEL
    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(Operation.REBOOT.host_status, "Reboot hasn't helped", "Host failed to reboot"),
            action=escalate_to_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(
                Operation.PROFILE.host_status, "Profiling hasn't helped", "Host profiling failed"
            ),
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=limit_reached("max_host_reboots", Operation.REBOOT),
            action=escalate_to_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            # we actually need "Operation.REPAIR_TAINTED_KERNEL" here, but there is no such thing
            reason=limit_reached("max_host_tainted_kernel_reboots", Operation.REBOOT),
            action=escalate_to_highload_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=automatic_profile_not_supported,
            action=escalate_to_deactivate,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_host_profiles", Operation.PROFILE, params={"modes": ProfileMode.HIGHLOAD_TEST}),
            action=escalate_to_deactivate,
        ),
    )

    def __init__(self, enabled=True, **kwargs):
        self.enabled = enabled

    def apply(self, host, check_result):
        if not self.enabled:
            return Decision.healthy("Host is healthy.")
        reasons = flatten_reason(check_result["metadata"]["result"]["reason"])
        return Decision(
            WalleAction.REBOOT,
            "Kernel on host has been tainted: {}.".format(reasons),
            checks=[CheckType.TAINTED_KERNEL],
            failure_type=FailureType.KERNEL_TAINTED,
            failure_check_info=check_result,
        )


class CheckCpu(CheckRuleInterface):
    check_type = CheckType.CPU
    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_CPU),
            reason=limit_reached("max_repaired_cpu", Operation.REPAIR_CPU),
            action=escalate_to_deactivate,
        )
    )

    def apply(self, host, check_result):
        reasons = flatten_reason(check_result["metadata"]["result"]["reason"])
        # TODO: use WalleAction.PROFILE for REPAIR_CPU, because it is actually highload profile.
        # repair cpu is currently the same action as profile, It has a separate decision mnemonic
        # because we need to apply different restrictions and limits.
        return Decision(
            WalleAction.REPAIR_CPU,
            "Problems with CPU detected: {}.".format(reasons),
            checks=[CheckType.CPU],
            failure_type=FailureType.CPU_FAILURE,
            failure_check_info=check_result,
        )


def _escalate_to_repair_hardware_for_capping(decision, reason):
    return decision.escalate(
        WalleAction.REPAIR_HARDWARE,
        reason,
        params=repair_hardware_params(
            request_type=RequestTypes.CPU_CAPPED.type, operation_type=Operation.REPAIR_CAPPING.type, reboot=True
        ),
        restrictions=[host_restrictions.AUTOMATED_CAPPING_REPAIR],
        failure_type=FailureType.CPU_CAPPED,
    )


class CheckCpuCapping(CheckRuleInterface):
    check_type = CheckType.CPU_CAPPING
    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REBOOT),
            reason=task_has_not_helped(Operation.REBOOT.host_status, "Reboot hasn't helped", "Host failed to reboot"),
            action=_escalate_to_repair_hardware_for_capping,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_cappings", Operation.REPAIR_CAPPING),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        reason = "CPU capping detected: {}.".format(check_result["metadata"]["reason"])
        return Decision(
            WalleAction.REBOOT,
            reason=reason,
            checks=[CheckType.CPU_CAPPING],
            restrictions=[host_restrictions.AUTOMATED_CAPPING_REPAIR],
            failure_type=FailureType.CPU_CAPPED,
            failure_check_info=check_result,
        )


class CheckReboots(CheckRuleInterface):
    check_type = CheckType.REBOOTS
    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_REBOOTS),
            reason=limit_reached("max_repaired_reboots", Operation.REPAIR_REBOOTS),
            action=escalate_to_second_time_node_report,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_REBOOTS),
            reason=limit_reached(
                "max_host_extra_highload_profiles", Operation.PROFILE, params={"modes": ProfileMode.EXTRA_HIGHLOAD_TEST}
            ),
            action=escalate_to_second_time_node_report,
        ),
    )

    def apply(self, host, check_result):
        return Decision(
            WalleAction.REPAIR_REBOOTS,
            "Host is rebooting too often.",
            checks=[CheckType.REBOOTS],
            failure_type=FailureType.MANY_REBOOTS,
            failure_check_info=check_result,
        )


def _escalate_to_disk_rw_test(decision, reason):
    return decision.escalate(
        WalleAction.PROFILE,
        reason,
        params={"profile_mode": ProfileMode.DISK_RW_TEST},
        restrictions=[
            host_restrictions.AUTOMATED_REDEPLOY,
            host_restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
            host_restrictions.AUTOMATED_PROFILE,
        ],
    )


class FsckRule(CheckRuleInterface):
    check_type = CheckType.FS_CHECK

    def __init__(self, enabled=True, **kwargs):
        self.enabled = enabled

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REDEPLOY),
            reason=task_has_not_helped(
                Operation.REDEPLOY.host_status, "Redeployment hasn't helped", "Host failed to redeploy"
            ),
            action=_escalate_to_disk_rw_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.REDEPLOY),
            reason=limit_reached("max_host_redeployments", Operation.REDEPLOY),
            action=_escalate_to_disk_rw_test,
        ),
        EscalationPoint(
            predicate=action_match(WalleAction.PROFILE),
            reason=limit_reached("max_host_profiles", Operation.PROFILE),
            action=escalate_to_deactivate,
        ),
    )

    def apply(self, host, check_result):
        if not self.enabled:
            return Decision.healthy("Host is healthy.")
        broken_partitions = []
        for device in check_result["metadata"]["result"]["device_list"]:
            if "status" in device:
                if device["status"] != "ok":
                    broken_partitions.append(device["message"])
            elif device["error_count"] > device["threshold"]:
                broken_partitions.append(device["message"])

        if len(broken_partitions) == 1:
            reason = "Filesystem is broken: {}.".format(broken_partitions[0])
        else:
            reason = "\n * ".join(["There are broken filesystems:"] + broken_partitions)

        return Decision(
            WalleAction.REDEPLOY,
            reason,
            checks=[CheckType.FS_CHECK],
            failure_type=FailureType.FS_CHECK,
            failure_check_info=check_result,
        )


class TorLinkRule(CheckRuleInterface):
    check_type = CheckType.TOR_LINK

    def __init__(self, enabled=True, **kwargs):
        # TODO(rocco66): remove it? always true?
        self.enabled = enabled

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_links", Operation.REPAIR_LINK),
            action=escalate_to_deactivate,
        )
    )

    def apply(self, host, check_result):
        if not self.enabled:
            return Decision.healthy("Host is available.")

        reason = "Errors/Drops detected on uplink ToR port"

        params = repair_hardware_params(
            request_type=RequestTypes.MALFUNCTIONING_LINK_RX_CRC_ERRORS.type,
            operation_type=Operation.REPAIR_LINK.type,
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params=params,
            checks=[CheckType.TOR_LINK],
            failure_type=FailureType.LINK_MALFUNCTION,
            restrictions=[host_restrictions.AUTOMATED_LINK_REPAIR],
            failure_check_info=check_result,
        )
