"""Rules for hw_watcher infiniband check."""

from walle import restrictions as host_restrictions
from walle.admin_requests.constants import RequestTypes
from walle.expert.decision import Decision
from walle.expert.failure_types import FailureType
from walle.expert.rules.base import CheckRuleInterface
from walle.expert.rules.escalation import (
    EscalationRules,
    EscalationPoint,
    action_match,
    limit_reached,
    escalate_to_deactivate,
)
from walle.expert.rules.utils import repair_hardware_params
from walle.expert.types import WalleAction, CheckType
from walle.operations_log.constants import Operation


class IbLinkRule(CheckRuleInterface):
    check_type = CheckType.IB_LINK

    escalation_rules = EscalationRules(
        EscalationPoint(
            predicate=action_match(WalleAction.REPAIR_HARDWARE),
            reason=limit_reached("max_repaired_ib_links", Operation.REPAIR_IB_LINK),
            action=escalate_to_deactivate,
        )
    )

    def apply(self, host, check_result):
        reason = "Errors detected on uplink infiniband ports: " + ", ".join(
            c["port"] for c in check_result["metadata"]["failed_checks"]
        )

        params = repair_hardware_params(
            request_type=RequestTypes.MALFUNCTIONING_LINK_RX_CRC_ERRORS.type,
            operation_type=Operation.REPAIR_IB_LINK.type,
        )
        return Decision(
            WalleAction.REPAIR_HARDWARE,
            reason,
            params=params,
            checks=[CheckType.IB_LINK],
            failure_type=FailureType.LINK_MALFUNCTION,
            restrictions=[host_restrictions.AUTOMATED_LINK_REPAIR, host_restrictions.AUTOMATED_INFINIBAND_REPAIR],
            failure_check_info=check_result,
        )
