"""Validation schemes for checks metadata"""

import typing

import fastjsonschema

from walle.expert.types import CheckType, CheckStatus, HwWatcherCheckStatus
from walle.util.jsonschema import OneOf, AnyOf, Array, Enum, Integer, Null, Number, Object, String
from walle.util.misc import drop_none


def _generate_metadata_validators() -> dict[str, dict[str, tuple[typing.Any]]]:
    netmon_metadata_schema = Object(
        properties={
            "status": Enum(
                elements=[CheckStatus.PASSED, CheckStatus.FAILED, CheckStatus.SUSPECTED, CheckStatus.MISSING]
            ),
            "connectivity": Object(
                properties={
                    "status": Enum(
                        elements=[CheckStatus.PASSED, CheckStatus.FAILED, CheckStatus.SUSPECTED, CheckStatus.MISSING]
                    ),
                    "alive": Number(),
                    "connectivity": Number(minimum=0, maximum=1),
                    "reason": String(min_length=1),
                },
                required=["status", "alive", "reason"],
            ),
            "volume": Object(
                properties={
                    "status": Enum(
                        elements=[CheckStatus.PASSED, CheckStatus.FAILED, CheckStatus.SUSPECTED, CheckStatus.MISSING]
                    ),
                    "reason": String(min_length=1),
                }
            ),
        },
        required=["status", "connectivity"],
    )

    memory_check_results_schema = {
        "ecc": AnyOf(
            schemes=[
                Object(
                    properties={
                        "status": Enum(elements=[HwWatcherCheckStatus.FAILED, HwWatcherCheckStatus.UNKNOWN]),
                        "slot": String(min_length=1),
                        "reason": Array(items=String(min_length=1), min_items=1),
                        "comment": String(min_length=1),
                        "timestamp": Number(),
                        "eine_code": Array(items=String(min_length=1)),
                    },
                    required=["status", "slot", "reason", "comment", "timestamp"],
                ),
                # this allows to pass checks in any status except FAILED and UNKNOWN
                Object(
                    properties={
                        "status": String(
                            pattern=r"^(?!{}|{})\w+$".format(HwWatcherCheckStatus.FAILED, HwWatcherCheckStatus.UNKNOWN)
                        ),
                        "timestamp": Number(),
                        "reason": Array(items=String(min_length=1)),
                    },
                    required=["status", "timestamp"],
                ),
            ]
        ),
        "mem": AnyOf(
            schemes=[
                # For "size" reason, also old hw-watcher did not have the "size" prefix, so we can not restrict it here.
                Object(
                    properties={
                        "status": Enum(elements=[HwWatcherCheckStatus.FAILED]),
                        "reason": Array(items=String(min_length=1), min_items=1),
                        "comment": String(min_length=1),
                        "needed_mem": Integer(minimum=1),
                        "real_mem": Integer(minimum=1),
                        "timestamp": Number(),
                        "eine_code": Array(items=String(min_length=1)),
                    },
                    required=["status", "reason", "comment", "needed_mem", "real_mem", "timestamp"],
                ),
                # For "numa" and "speed" reasons
                Object(
                    properties={
                        "status": Enum(elements=[HwWatcherCheckStatus.FAILED]),
                        "reason": Array(items=String(min_length=1, pattern=r"^(numa|speed): "), min_items=1),
                        "timestamp": Number(),
                        "eine_code": Array(items=String(min_length=1)),
                    },
                    required=["status", "reason", "timestamp"],
                ),
                # this allows to pass checks in any status except FAILED
                Object(
                    properties={
                        "status": String(pattern=r"^(?!{})\w+$".format(HwWatcherCheckStatus.FAILED)),
                        "timestamp": Number(),
                        "reason": Array(items=String(min_length=1)),
                    },
                    required=["status", "timestamp"],
                ),
            ]
        ),
    }

    common_schema = Object(
        properties={
            "result": Object(
                properties={
                    "status": Enum(elements=[HwWatcherCheckStatus.FAILED]),
                    "reason": Array(items=String(min_length=1), min_items=1),
                    "timestamp": Number(),
                    "eine_code": Array(items=String(min_length=1)),
                },
                required=["status", "reason", "timestamp"],
            ),
        },
        required=["result"],
    )

    rack_ok_schema = Object(
        properties={
            "total": Integer(minimum=0),
            "minimal_amount": Integer(minimum=0),
            "suspected": Integer(minimum=0),
            "failed": Integer(minimum=0),
            "threshold_flapping": Integer(minimum=0),
            "threshold_failed": Integer(minimum=0),
            "last_crit": Integer(minimum=0),
            "suspected_timeout": Integer(minimum=0),
        },
        required=["total"],
    )

    rack_failed_schema = Object(
        properties={
            "total": Integer(minimum=0),
            "suspected": Integer(minimum=0),
            "failed": Integer(minimum=0),
            "threshold_flapping": Integer(minimum=0),
            "threshold_failed": Integer(minimum=0),
            "last_crit": Integer(minimum=0),
            "suspected_timeout": Integer(minimum=0),
        },
        required=["total", "suspected", "failed", "threshold_flapping", "threshold_failed", "suspected_timeout"],
    )

    def rack_preprocessor(metadata: dict[str:int]) -> dict[str, int]:
        return drop_none(metadata)

    def disk2replace_preprocessor(metadata: dict[str, typing.Any]) -> dict[str, typing.Any]:
        if "result" in metadata and "disk2replace" in metadata["result"]:
            metadata["result"]["disk2replace"] = drop_none(metadata["result"]["disk2replace"])
        return metadata

    schemes = [
        ((CheckType.UNREACHABLE, CheckType.SSH, CheckType.META), CheckStatus.ALL_JUGGLER, String(min_length=2), None),
        (
            CheckType.W_META,
            CheckStatus.ALL_JUGGLER,
            Object(
                properties={
                    "bundle": Number(),
                    "timestamp": Number(),
                    "hw_watcher": String(),
                },
                required=["bundle", "timestamp", "hw_watcher"],
            ),
            None,
        ),
        (
            CheckType.MEMORY,
            CheckStatus.FAILED,
            Object(
                properties={
                    "results": Object(properties=memory_check_results_schema, required=["ecc", "mem"]),
                },
                required=["results"],
            ),
            None,
        ),
        (
            CheckType.MEMORY,
            (CheckStatus.PASSED, CheckStatus.SUSPECTED),
            OneOf(
                schemes=[
                    Object(
                        properties={
                            "reason": String(min_length=1),
                        },
                        required=["reason"],
                    ),
                    Object(
                        properties={"results": Object(properties=memory_check_results_schema, required=["ecc", "mem"])},
                        required=["results"],
                    ),
                ]
            ),
            None,
        ),
        (
            CheckType.DISK,
            CheckStatus.FAILED,
            OneOf(
                schemes=[
                    Object(
                        properties={
                            "reason": String(min_length=1),
                        },
                        required=["reason"],
                    ),
                    Object(
                        properties={
                            "result": Object(
                                properties={
                                    "status": Enum(
                                        elements=[HwWatcherCheckStatus.FAILED, HwWatcherCheckStatus.UNKNOWN]
                                    ),
                                    "reason": Array(items=String(min_length=1), min_items=1),
                                    "disk2replace": Object(
                                        properties={
                                            "slot": OneOf(schemes=[Integer(minimum=-1), String(min_length=1)]),
                                            "disk_type": String(min_length=1),
                                            "model": String(min_length=1),
                                            "serial": String(min_length=1),
                                            "shelf_inv": String(min_length=1),
                                            "diskperformance": String(min_length=1),
                                            "reason": Array(items=String(min_length=1)),
                                        },
                                        required=["slot", "reason"],
                                    ),
                                    "timestamp": Number(),
                                    "eine_code": Array(items=String(min_length=1)),
                                },
                                required=["status", "reason", "timestamp"],
                            ),
                        },
                        required=["result"],
                    ),
                ]
            ),
            disk2replace_preprocessor,
        ),
        (
            CheckType.LINK,
            CheckStatus.FAILED,
            Object(
                properties={
                    "result": Object(
                        properties={
                            "reason": Array(items=String(min_length=1), min_items=1),
                            "timestamp": Number(),
                            "eine_code": Array(items=String(min_length=1)),
                        },
                        required=["reason", "timestamp"],
                    )
                },
                required=["result"],
            ),
            None,
        ),
        (CheckType.BMC, CheckStatus.FAILED, common_schema, None),
        (CheckType.GPU, CheckStatus.FAILED, common_schema, None),
        (CheckType.INFINIBAND, CheckStatus.FAILED, common_schema, None),
        (
            CheckType.CPU_CACHES,
            CheckStatus.FAILED,
            Object(
                properties={
                    "result": AnyOf(
                        schemes=[
                            Object(
                                properties={
                                    "socket": Null(),
                                    "status": Enum(elements=[HwWatcherCheckStatus.FAILED]),
                                    "reason": Array(items=String(min_length=1), min_items=1),
                                    "raw": String(),
                                    "timestamp": Number(),
                                    "eine_code": Array(
                                        items=Enum(elements=["CPU_OVERHEATING", "SYSTEM_MISCONFIGURED"])
                                    ),
                                },
                                required=["status", "reason", "timestamp"],
                            ),
                            Object(
                                properties={
                                    "socket": Integer(minimum=0),
                                    "status": Enum(elements=[HwWatcherCheckStatus.FAILED]),
                                    "reason": Array(items=String(min_length=1), min_items=1),
                                    "raw": String(),
                                    "timestamp": Number(),
                                    "eine_code": Array(items=String(min_length=1)),
                                },
                                required=["socket", "status", "reason", "timestamp"],
                            ),
                        ]
                    ),
                },
                required=["result"],
            ),
            None,
        ),
        (
            (CheckType.LINK, CheckType.DISK, CheckType.BMC, CheckType.GPU, CheckType.INFINIBAND, CheckType.CPU_CACHES),
            (CheckStatus.PASSED, CheckStatus.SUSPECTED),
            Object(
                properties={"result": Object(properties={"timestamp": Number()}, required=["timestamp"])}, required=[]
            ),
            None,
        ),
        (
            CheckType.REBOOTS,
            CheckStatus.ALL_JUGGLER,
            Object(
                properties={"result": Object(properties={"count": Integer(minimum=0)}, required=["count"])},
                required=["result"],
            ),
            None,
        ),
        (
            (CheckType.TAINTED_KERNEL, CheckType.CPU),
            CheckStatus.ALL_JUGGLER,
            Object(
                properties={
                    "result": Object(
                        properties={"reason": Array(items=String(min_length=1), min_items=1)}, required=["reason"]
                    ),
                },
                required=["result"],
            ),
            None,
        ),
        (
            CheckType.FS_CHECK,
            CheckStatus.ALL_JUGGLER,
            Object(
                properties={
                    "result": Object(
                        properties={
                            "device_list": Array(
                                items=Object(
                                    properties={
                                        "name": String(min_length=1),
                                        "error_count": Integer(minimum=-1),
                                        "threshold": Integer(minimum=0),
                                        "message": String(min_length=1),
                                        "status": Enum(elements=["ok", "suspected", "failed"]),
                                    },
                                    required=["name", "error_count", "threshold", "message"],
                                )
                            )
                        },
                        required=["device_list"],
                    )
                },
                required=["result"],
            ),
            None,
        ),
        (CheckType.WALLE_RACK, CheckStatus.PASSED, rack_ok_schema, rack_preprocessor),
        (CheckType.WALLE_RACK, (CheckStatus.FAILED, CheckStatus.SUSPECTED), rack_failed_schema, rack_preprocessor),
        (CheckType.WALLE_RACK_OVERHEAT, CheckStatus.PASSED, rack_ok_schema, rack_preprocessor),
        (
            CheckType.WALLE_RACK_OVERHEAT,
            (CheckStatus.FAILED, CheckStatus.SUSPECTED),
            rack_failed_schema,
            rack_preprocessor,
        ),
        (
            CheckType.NETMON,
            CheckStatus.ALL_JUGGLER + [CheckStatus.MISSING],
            Object(
                properties={
                    "datacenter": netmon_metadata_schema,
                    "queue": netmon_metadata_schema,
                    "switch": netmon_metadata_schema,
                    "timestamp": Number(),
                },
                required=["datacenter", "queue", "switch", "timestamp"],
            ),
            None,
        ),
    ]

    schema_mapping = {}
    preprocessor_mapping = {}

    for check_types, check_statuses, schema, preprocessor in schemes:
        if isinstance(check_types, str):
            check_types = (check_types,)

        if isinstance(check_statuses, str):
            check_statuses = (check_statuses,)

        for check_type in check_types:
            for check_status in check_statuses:
                if schema_mapping.setdefault(check_type, {}).setdefault(check_status, schema) is not schema:
                    raise ValueError("Failed to generate metadata schemes: there is overlapping by check statuses.")
                preprocessor_mapping.setdefault(check_type, {})[check_status] = preprocessor

    validators_mapping = {}

    for check_type, check_type_schemes in schema_mapping.items():
        for check_status, schema in check_type_schemes.items():
            validators_mapping.setdefault(check_type, {})[check_status] = (
                fastjsonschema.compile(schema.get_schema()),
                preprocessor_mapping[check_type][check_status],
            )

    return validators_mapping


_DEFAULT_METADATA_SCHEME = Object(
    properties={
        "reason": String(min_length=0),
        "timestamp": Number(),
    },
    required=["reason", "timestamp"],
)

METADATA_VALIDATORS = _generate_metadata_validators()
DEFAULT_METADATA_VALIDATOR = fastjsonschema.compile(_DEFAULT_METADATA_SCHEME.get_schema()), None
