from walle.util.misc import values_ordered


class CheckType:
    UNREACHABLE = "UNREACHABLE"
    SSH = "ssh"
    ALL_ACTIVE = [UNREACHABLE, SSH]
    """Juggler active checks."""

    META = "META"
    W_META = "walle_meta"
    MEMORY = "walle_memory"
    DISK = "walle_disk"
    LINK = "walle_link"
    BMC = "walle_bmc"
    GPU = "walle_gpu"
    INFINIBAND = "walle_infiniband"
    REBOOTS = "walle_reboots"
    TAINTED_KERNEL = "walle_tainted_kernel"
    CPU = "walle_cpu"
    CPU_CACHES = "walle_cpu_caches"
    CPU_CAPPING = "walle_cpu_capping"
    WALLE_RACK_OVERHEAT = "walle_rack_overheat"
    FS_CHECK = "walle_fs_check"
    CLOCKSOURCE = "walle_clocksource"
    TOR_LINK = "walle_tor_link"
    IB_LINK = "walle_ib_link"

    ALL_HW_WATCHER = [MEMORY, DISK, LINK, BMC, GPU, CPU_CACHES, INFINIBAND]
    """Checks provided by hw-watcher for our modern automation plot."""

    ALL_IB = [INFINIBAND, IB_LINK]
    """All infiniband checks (hw-watcher and linkeye)"""

    OPTIONAL_HW_WATCHER = [CLOCKSOURCE]
    """Checks provided by hw-watcher but not in default automation plot"""

    ALL_PASSIVE = [
        META,
        W_META,
        REBOOTS,
        TAINTED_KERNEL,
        CPU,
        CPU_CAPPING,
        FS_CHECK,
        TOR_LINK,
        IB_LINK,
    ] + ALL_HW_WATCHER
    """Juggler passive checks we use in our modern automation plot."""

    ALL_HARDWARE = ALL_HW_WATCHER + [CPU, CPU_CAPPING, WALLE_RACK_OVERHEAT, IB_LINK]
    """Checks that should go to the 'hardware' group in UI."""

    ALL_META = [META, W_META]
    """Meta-checks that only check that checks are working."""

    ALL_SOFTWARE = [META, W_META, REBOOTS, TAINTED_KERNEL, FS_CHECK]
    """Checks that should go to the 'software' group in UI."""

    ALL_JUGGLER = ALL_ACTIVE + ALL_PASSIVE
    """Checks produced by Juggler for our modern automation plot."""

    NETMON = "netmon"
    ALL_NETMON = [NETMON]
    """Checks based on netmon data."""

    WALLE_RACK = "walle_rack"

    ALL_WALLE = [WALLE_RACK, WALLE_RACK_OVERHEAT]
    """Meta-checks generated ny Wall-E."""

    ALL_RACK = [WALLE_RACK, WALLE_RACK_OVERHEAT]
    """All rack type checks"""

    ALL_AVAILABILITY = [UNREACHABLE, SSH]
    """Checks that test host's availability"""

    ALL_INFRASTRUCTURE = [WALLE_RACK]
    """Checks that test network or other infrastructure."""
    # TODO: WALLE-4639 - We decided to remove netmon from infra checks, it's too unstable

    ALL = ALL_JUGGLER + ALL_NETMON + ALL_WALLE
    """A list of supported Expert check types."""

    WALLE_MAPPING = {
        UNREACHABLE: "unreachable",
        SSH: "ssh",
        META: "meta",
        W_META: "walle_meta",
        MEMORY: "memory",
        DISK: "disk",
        LINK: "link",
        REBOOTS: "reboots",
        TAINTED_KERNEL: "tainted_kernel",
        CPU: "cpu",
        CPU_CACHES: "cpu_caches",
        CPU_CAPPING: "cpu_capping",
        FS_CHECK: "fs_check",
        BMC: "bmc",
        GPU: "gpu",
        INFINIBAND: "infiniband",
        WALLE_RACK: "rack",
        WALLE_RACK_OVERHEAT: "rack_overheat",
        NETMON: "switch",
        TOR_LINK: "tor_link",
        IB_LINK: "ib_link",
    }
    """Maps Juggler check types to Wall-E check types for UI."""

    ALL_UI_TYPES = values_ordered(WALLE_MAPPING, ALL)
    """This list is exposed to the constants API and used in host health filters."""

    UI_TYPE_GROUPS = {
        "availability": values_ordered(WALLE_MAPPING, ALL_AVAILABILITY),
        "infrastructure": values_ordered(WALLE_MAPPING, ALL_INFRASTRUCTURE + [NETMON]),
        "hardware": values_ordered(WALLE_MAPPING, ALL_HARDWARE + [TOR_LINK]),
        "software": values_ordered(WALLE_MAPPING, ALL_SOFTWARE),
    }
    """This list is exposed to the constants API and used to display host health status."""

    ALL_DMC_TYPES = (
        [
            REBOOTS,
            TAINTED_KERNEL,
            CPU,
            CPU_CAPPING,
            FS_CHECK,
            W_META,
            TOR_LINK,
            IB_LINK,
        ]
        + ALL_RACK
        + ALL_HW_WATCHER
        + ALL_AVAILABILITY
    )
    """List of checks that can possibly appear in a DMC Decision object. Checks that Wall-E can try to fix."""

    assert set(WALLE_MAPPING) == set(ALL)


def get_walle_check_type(check_type):
    """Maps Expert System's check types to Wall-E check types."""

    # TODO(rocco66): different types for different checks
    try:
        return CheckType.WALLE_MAPPING[check_type]
    except KeyError:
        # project's custom check. may be transform it somehow
        return check_type


class CheckGroup:
    """Represents groups of checks that should be checked to check host for a specific error types."""

    NETWORK_AVAILABILITY = CheckType.ALL_AVAILABILITY + CheckType.ALL_INFRASTRUCTURE
    OS_CONSISTENCY = [CheckType.REBOOTS, CheckType.TAINTED_KERNEL]


class CheckStatus:
    PASSED = "passed"
    """Check passed."""

    SUSPECTED = "suspected"
    """Host marked suspected because check failed for host but it's to early to mark it failed."""

    FAILED = "failed"
    """Host marked failed because check failed."""

    MISSING = "missing"
    """Check data is missing - Wall-E has received a NO DATA check."""

    VOID = "void"
    """Check is missing, Wall-E has not received check data."""

    INVALID = "invalid"
    """Check has invalid metadata."""

    STALED = "staled"
    """Check is staled."""

    UNSURE = "unsure"
    """Check data is not fresh enough, need to wait for check status update."""

    ALL_JUGGLER = [FAILED, PASSED, SUSPECTED]
    """A list of possible Expert check statuses."""

    ALL_IGNORED = {SUSPECTED, STALED}
    """We ignore checks in these statuses."""

    ALL_MISSING = [MISSING, VOID, STALED]
    """Merge these statuses into one metric."""

    ALL_WITHOUT_METADATA = {MISSING, INVALID, VOID}
    """Checks in these statuses may not have metadata."""

    ALL_YASM_DMC_MONITORED = [INVALID, MISSING, VOID, STALED]
    """These statues used to render yasm panel and alert templates."""

    ALL = ALL_JUGGLER + [MISSING, INVALID, VOID, STALED, UNSURE]
    """A list of possible check statuses from Wall-E's point of view."""


ALL_STATUSES = {"OK": CheckStatus.PASSED, "WARN": CheckStatus.SUSPECTED, "CRIT": CheckStatus.FAILED}


class HwWatcherCheckStatus:
    FAILED = "FAILED"
    UNKNOWN = "UNKNOWN"


_BASIC_MODE_CHECKS = CheckType.ALL_ACTIVE + [CheckType.META] + [CheckType.WALLE_RACK]
_FULL_FEATURED_MODE_CHECKS = CheckType.ALL_ACTIVE + [
    CheckType.W_META,
    CheckType.LINK,
    CheckType.MEMORY,
    CheckType.DISK,
    CheckType.BMC,
    CheckType.GPU,
    CheckType.INFINIBAND,
    CheckType.REBOOTS,
    CheckType.TAINTED_KERNEL,
    CheckType.CPU,
    CheckType.CPU_CACHES,
    CheckType.CPU_CAPPING,
    CheckType.FS_CHECK,
    CheckType.TOR_LINK,
    CheckType.IB_LINK,
]


class CheckSets:
    BASIC = frozenset(CheckType.ALL_ACTIVE + _BASIC_MODE_CHECKS + CheckType.ALL_NETMON)

    FULL_FEATURED = frozenset(
        CheckType.ALL_ACTIVE + _FULL_FEATURED_MODE_CHECKS + CheckType.ALL_NETMON + CheckType.ALL_WALLE
    )


class WalleAction:
    WAIT = "wait"
    HEALTHY = "healthy"
    FAILURE = "failure"

    REBOOT = "reboot"
    PROFILE = "profile"
    REDEPLOY = "redeploy"
    REPORT_FAILURE = "report-failure"

    DISK_RW_REDEPLOY = "disk-rw-test-and-redeploy"

    REPAIR_HARDWARE = "repair-hardware"
    CHANGE_DISK = "change-disk"
    REPAIR_CPU = "repair-cpu"
    RESET_BMC = "reset-bmc"
    REPAIR_REBOOTS = "repair-reboots"
    REPAIR_MEMORY = "repair-memory"

    REPAIR_RACK_FAILURE = "repair-rack-failure"
    REPAIR_RACK_OVERHEAT = "repair-rack-overheat"

    # TODO: drop this action altogether with the DEAD status
    DEACTIVATE = "deactivate"

    FIX_DNS = "fix-dns"

    ALL_DMC = [
        WAIT,
        HEALTHY,
        DEACTIVATE,
        REBOOT,
        PROFILE,
        REDEPLOY,
        REPORT_FAILURE,
        CHANGE_DISK,
        REPAIR_CPU,
        REPAIR_HARDWARE,
        RESET_BMC,
        REPAIR_MEMORY,
        REPAIR_REBOOTS,
        REPAIR_RACK_FAILURE,
        REPAIR_RACK_OVERHEAT,
    ]
    """A list of actions that are supported by DMC failure handling mechanism."""

    ALL_CRON = [FIX_DNS]
    """A list of actions that are handled by separated cron daemons apart from DMC failure handling mechanism."""

    ALL_MONITOR = ALL_DMC + [FAILURE]
    """List of decision actions used in the monitor stage (e.g. for non-automated tasks)."""

    ALL = ALL_CRON + ALL_MONITOR
    """A list of all possible actions."""


class Failure:
    """All check failures are failures. On top of that we've got these known failure types."""

    CHECKS_MISSING = "checks_missing"

    DEACTIVATE = WalleAction.DEACTIVATE
    FIX_DNS = WalleAction.FIX_DNS

    ALL_FAILURE_CHECKS = list(map(get_walle_check_type, CheckType.ALL_DMC_TYPES))
    """All DMC check types can be used as failures."""

    ALL_FAILURE_ACTIONS = [FIX_DNS, DEACTIVATE]
    """List of actions that should be registered in FailureLog."""

    ALL = ALL_FAILURE_CHECKS + [CHECKS_MISSING, FIX_DNS, DEACTIVATE]
    """All known failure types"""

    ALL_RACK = list(map(get_walle_check_type, CheckType.ALL_RACK))
    """All rack types"""


def _mk_limit_name(failure):
    return "max_{}_failures".format(failure)


def _mk_failure_limit_map():
    names_map = {failure: _mk_limit_name(failure) for failure in Failure.ALL}
    names_map.update(
        {
            Failure.FIX_DNS: "max_dns_fixes",
            Failure.DEACTIVATE: "max_dead_hosts",
        }
    )

    return names_map


FAILURE_LIMITS_MAP = _mk_failure_limit_map()


def get_limit_name(failure):
    try:
        return FAILURE_LIMITS_MAP[failure]
    except KeyError:
        return _mk_limit_name(failure)
