"""Collect base report abstraction here."""

import datetime
import itertools
import logging
import uuid
from collections import namedtuple
from operator import attrgetter

from mongoengine import fields as db_fields, EmbeddedDocument

from sepelib.core import config
from sepelib.core.exceptions import Error, LogicalError
from sepelib.mongo.util import register_model
from walle.hosts import Host
from walle.models import Document, timestamp
from walle.util.misc import drop_none
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


class ReportedHost(EmbeddedDocument):
    inv = db_fields.IntField(min_value=0, required=True, help_text="Host inventory number")
    name = db_fields.StringField(default=None, help_text="Name of host when it was reported")
    host_uuid = db_fields.StringField(default=None, help_text="UUID of host when it was reported")
    project = db_fields.StringField(default=None, help_text="Project of host when it was reported")
    status = db_fields.StringField(default=None, help_text="Status of host when it was reported")
    section = db_fields.StringField(default=None, help_text="Name of report section in which host was reported")
    reason = db_fields.StringField(default=None, help_text="Optional host failure reason")
    solved = db_fields.BooleanField(default=None, help_text="Indicate that host's problem has been resolved")
    tickets = db_fields.ListField(db_fields.StringField(), default=None, help_text="Tickets related to this failure")
    audit_log_id = db_fields.StringField(default=None, help_text="A link to audit log entry")
    report_timestamp = db_fields.LongField(help_text="Timestamp of first appearance in report")
    solve_timestamp = db_fields.LongField(help_text="Host resolve timestamp")

    def __unicode__(self):
        return self.to_json()

    def to_dict(self):
        return drop_none(
            dict(
                inv=self.inv,
                name=self.name,
                uuid=self.host_uuid,
                status=self.status,
                project=self.project,
                audit_log_id=self.audit_log_id,
                tickets=self.tickets,
                reason=self.reason,
                solved=self.solved,
                section=self.section,
            )
        )

    def update(self, new_data):
        for key, value in new_data.items():
            setattr(self, key, value)


@register_model
class ErrorReportModel(Document):
    report_key = db_fields.StringField(primary_key=True, help_text="Report's unique key")
    stream_key = db_fields.StringField(required=True, help_text="Internal key for unique set of report parameters")
    # We only need this because we create new report every day and forcefully close previous report.
    report_date = db_fields.DateTimeField(required=True, help_text="Date for which this report was created")
    create_time = db_fields.LongField(required=True, help_text="Report create timestamp")
    last_update_time = db_fields.LongField(required=True, help_text="Report last update timestamp")
    closed = db_fields.BooleanField(default=None, help_text="Indicate a closed report")
    hosts = db_fields.ListField(db_fields.EmbeddedDocumentField(ReportedHost), required=True)

    meta = {
        "collection": "error_reports",
    }

    @classmethod
    def create(cls, report_key, stream_key, report_date, hosts):
        now = timestamp()
        cls(
            report_key=report_key,
            stream_key=stream_key,
            report_date=report_date,
            create_time=now,
            last_update_time=now,
            hosts=hosts,
        ).save(force_insert=True)

    @classmethod
    def update(cls, report_key, **kwargs):
        now = timestamp()
        cls.objects(report_key=report_key).modify(upsert=False, last_update_time=now, **kwargs)

    @classmethod
    def close_report(cls, report_key):
        cls.update(report_key, closed=True)

    @classmethod
    def find(cls, stream_key):
        """Return list of open reports from stream. Most recent report goes last."""
        return cls.objects(stream_key=stream_key, closed__ne=True).order_by("report_date", "create_time")

    @classmethod
    def opened(cls, last_update):
        """Return all opened reports older than provided last update time."""
        return cls.objects(closed__ne=True, last_update_time__lte=last_update).order_by("report_date", "create_time")


def now():
    # N.B. this delegation is used in tests for monkeypatching
    return datetime.datetime.fromtimestamp(timestamp())


def relative_today():
    # Open new ticket at 9:00 am MSK. Maintain previous ticket until then. Effectively, new day starts at 9:00.
    # There may be weird corner cases, like when wall-e creates the very first report for the project at 4:00 am
    # and it's date then is yesterday, but I don't expect this to happen, and when it happen it is not a problem.
    return (now() - datetime.timedelta(hours=9)).date()


def _random_report_key():
    return str(uuid.uuid4())


class ReportFailure(Error):
    """Represent a failure to construct, save or update report."""

    pass


class ReportPublisherFailure(ReportFailure):
    """Represent a failure to save or update startrek ticket."""

    pass


class ReportPublisher:
    """Report publisher interface, used by ErrorHostsReport to publish reports to user."""

    name = None

    def get_stream_key(self):
        raise NotImplementedError

    @classmethod
    def from_stream_key(cls, stream_key):
        """Construct instance from a stream key"""
        raise NotImplementedError

    def close_old_report(self, report_key):
        raise NotImplementedError

    def create_new_report(self, summary, report_text, report_hosts, previous_report=None):
        raise NotImplementedError

    def update_existing_report(self, report, report_text, report_hosts):
        raise NotImplementedError

    def verify_report_published(self, report_key):
        raise NotImplementedError

    @staticmethod
    def by_stream_key(stream_key):
        """
        Find and construct publisher by stream key.

        :type stream_key: StreamKey
        """
        for publisher_cls in ReportPublisher.__subclasses__():
            if publisher_cls.name == stream_key.publisher_name:
                return publisher_cls.from_stream_key(stream_key.unwrapped_key())

        raise LogicalError  # publisher not registered.


class ReportObserver:
    """Report observer interface, used by ErrorHostsReport to perform additional actions when report changes."""

    def report_created(self, report_key, report_hosts, previous_report_key=None):
        pass

    def report_failed_to_create(self, error_message, report_hosts):
        pass

    def report_updated(self, report_key, report_hosts):
        pass

    def report_closed(self, report_key, report_hosts):
        pass


class StreamKey:
    # This object might be a Stream object and include methods like create, close, update, etc... But not today.
    def __init__(self, stream_key, publisher_name="startrek", rotation_strategy_name="daily", original_stream_key=None):
        self.stream_key = stream_key
        self.publisher_name = publisher_name
        self.rotation_strategy_name = rotation_strategy_name
        self.original_stream_key = original_stream_key

    def wrapped_key(self):
        if self.original_stream_key is not None:
            return self.original_stream_key

        return "{publisher}/{rotation}@{stream_key}".format(
            publisher=self.publisher_name, rotation=self.rotation_strategy_name, stream_key=self.stream_key
        )

    def unwrapped_key(self):
        return self.stream_key

    @classmethod
    def from_wrapped_key(cls, stream_key):

        parts = stream_key.split("@", 1)
        if len(parts) == 2:
            stream_params_str, unwrapped_key = parts
            stream_params = stream_params_str.split("/")
        else:
            [unwrapped_key] = parts
            stream_params = []

        return cls(unwrapped_key, *stream_params, original_stream_key=stream_key)


class ErrorHostsReport:
    """Manipulate report on Wall-E level: store it into database, maintain related issues, etc."""

    _current_report = None
    _previous_report = None
    _old_reports = None

    _stream_key = None
    _report_key = None
    _report_summary = None
    _report_content = None

    _publisher = None
    _dry_run = None

    def __init__(self, stream_key, publisher, rotation_strategy, observer=None, raise_on_failure=True, dry_run=False):
        """
        :param stream_key: publish reports to given stream key
        :param publisher: publish reports to given backend
        :param rotation_strategy: function that finds existing reports in the stream
        :param observer: report observer that can update host tickets or create audit log records
        :param dry_run: do not actually store report to the database or publish them.

        :type stream_key: unicode
        :type publisher: ReportPublisher
        :type rotation_strategy: RotationStrategy
        :type observer ReportObserver
        """
        self._stream_key = stream_key
        self._publisher = publisher
        self._observer = observer or ()
        self._rotation = rotation_strategy
        self._raise_on_failure = raise_on_failure
        self._dry_run = dry_run

    # "Public" api: report is a context manager which commits report on exit.
    # Client may provide a description for it, but also client may skip it if report is not going
    # to create a new issue (or may skip if report is only going to create a new issue).
    def is_new_report(self):
        """Return true if a new issue for this report will be created."""
        return self._current_report is None

    def set_content(self, summary, content):
        """User may set contents for the report if they want to proceed with this issue
        (whether a new or an existing one).
        Summary will only be applied to the new report if it is going to be created.

        :type summary: unicode
        :type content: list(ReportSection)

        """
        self._report_summary = summary
        self._report_content = content

    def get_report_key(self):
        return self._report_key

    def __enter__(self):
        existing_reports = self._rotation.find_reports(self._stream_key)
        self._current_report = existing_reports.current_report
        self._previous_report = existing_reports.previous_report
        self._old_reports = existing_reports.old_reports

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if not exc_type:
            try:
                self._close_old_reports()
                if self._report_content is not None:
                    self._make_report(self._report_content)
            except ReportFailure as e:
                if self._raise_on_failure:
                    raise
                # These error handled already, but they are not recoverable, go to next report
                log.error(e)

    def _make_report(self, report_content):
        """Create new report or update existing one."""
        if self.is_new_report():
            if not report_content.empty():
                self._create_new_report(report_content)
        else:
            self._proceed_with_existing_report(self._current_report, report_content)

    def _close_old_reports(self):
        for report in self._old_reports or ():
            self._close_old_report(report)

    def _create_new_report(self, report_content):
        """
        :param report_content: report content sections
        """
        if self._dry_run:
            log.debug("Store report with random key in stream %s", self._stream_key)
            return

        report_hosts = report_content.report_hosts()

        try:
            new_report_key = self._publisher.create_new_report(
                self._report_summary, report_content.text(), report_hosts, self._previous_report
            )
        except ReportFailure as e:
            self._observer.report_failed_to_create(str(e), report_hosts=report_hosts)
            raise
        else:
            self._observer.report_created(
                new_report_key,
                report_hosts=report_hosts,
                previous_report_key=self._previous_report.report_key if self._previous_report else None,
            )

        self._report_key = new_report_key

    def _proceed_with_existing_report(self, report, new_report_content):
        new_report_content.merge_hosts({host.inv: host for host in report.hosts})

        published = self._publisher.verify_report_published(report.report_key)

        if new_report_content.empty():
            self._close_old_report(report, new_report_content)
        elif published:
            self._update_existing_report(report, new_report_content)
        else:
            log.info("Report %s is not published. Creating and publishing a new one.", report.report_key)

            self._previous_report = report
            self._close_old_report(report)
            self._create_new_report(new_report_content)

    def _update_existing_report(self, report, new_report_content):
        report_hosts = new_report_content.report_hosts()

        self._publisher.update_existing_report(report, new_report_content.text(), report_hosts)
        self._observer.report_updated(report.report_key, report_hosts=report_hosts)

        self._report_key = report.report_key

    def _close_old_report(self, report, new_report_content=None):
        if self._dry_run:
            log.debug("Close report with key %s in stream %s", report.report_key, self._stream_key)
            return

        report_hosts = report.hosts

        if new_report_content:
            report_hosts = new_report_content.report_hosts()
            self._publisher.update_existing_report(report, new_report_content.text(), report_hosts)

        self._publisher.close_old_report(report.report_key)
        self._observer.report_closed(report.report_key, report_hosts=report_hosts)


ExistingReports = namedtuple("ExistingReports", ["current_report", "previous_report", "old_reports"])


class RotationStrategy:
    """Interface for report rotation function."""

    name = None

    def find_reports(self, stream_key):
        """
        Find open reports in reports collection by stream key and return an ExistingReports tuple.

        :type stream_key: unicode
        :rtype: ExistingReports
        """
        raise NotImplementedError

    @classmethod
    def by_name(cls, name):
        """Find and construct publisher by name obtained from a stream key."""
        for rotation_cls in (DailyReportRotationStrategy, HostFailureRotationStrategy):
            if rotation_cls.name == name:
                return rotation_cls()

        raise LogicalError  # publisher not registered.


class DailyReportRotationStrategy(RotationStrategy):
    name = "daily"

    def find_reports(self, stream_key):
        """Keep today's report open, close all previous reports."""

        today_report = None
        previous_report = None
        old_reports = []

        today = relative_today()

        for report in ErrorReportModel.find(stream_key):
            log.debug("Got existing report %s", report.report_key)
            if report.report_date.date() >= today:
                if today_report:
                    log.error(
                        "There are more then one report for today's date %s in stream %s. Closing duplicates.",
                        today.strftime("%Y-%m-%d"),
                        stream_key,
                    )
                    previous_report = today_report
                    old_reports.append(today_report)

                today_report = report
            else:
                # rely on ErrorReportModel.find returning reports in order, with most recent report returned last.
                previous_report = report
                old_reports.append(report)

        return ExistingReports(today_report, previous_report, old_reports)


class HostFailureRotationStrategy(RotationStrategy):
    name = "failing_hosts"

    def __init__(self, rotate=True):
        """
        Find open tickets in a stream and mark for garbage collecting tickets without failing hosts.
        Precisely, check if any of report's hosts have running tasks.
        Skip the check if rotate is False (used for creating reports from host's task).
        rotate defaults to True for garbage collecting (gc constructs strategies without arguments).
        """
        self._rotate = rotate

    def find_reports(self, stream_key):
        current_report = None
        previous_report = None
        old_reports = []

        for report in ErrorReportModel.find(stream_key):
            if self._rotate and self._is_staled(report):
                log.debug("Got staled report %s", report.report_key)
                previous_report = report
                old_reports.append(report)
                continue

            log.debug("Got existing report %s", report.report_key)
            if current_report:
                log.error("There are more then one report in stream %s. Closing duplicates.", stream_key)
                previous_report = current_report
                old_reports.append(current_report)

            current_report = report

        return ExistingReports(current_report, previous_report, old_reports)

    @staticmethod
    def _is_staled(report):
        report_hosts = [report_host.inv for report_host in report.hosts]
        if not report_hosts:
            return False

        collection = Host.get_collection(read_preference=SECONDARY_LOCAL_DC_PREFERRED)
        query = {"inv": {"$in": report_hosts}, "task": {"$exists": True}}
        return collection.find(query).count() == 0


class ReportSection:
    """Abstract report section have some capabilities to find error hosts."""

    def __init__(self, name, formatter):
        self.name = name
        self.formatter = formatter

        self.problem_hosts = {}  # type: dict[int, ReportedHost]
        self.solved_hosts = {}  # type: dict[int, ReportedHost]

    def add_problem_host(self, host):
        host.section = self.name
        self.problem_hosts[host.inv] = host

    def update_problem_host(self, stored_host):
        inv = stored_host.inv

        # take all stored data and apply new data on top of it
        stored_host.update(self.problem_hosts[inv].to_dict())
        self.problem_hosts[inv] = stored_host

    def add_solved_host(self, host):
        host.solved = True
        host.solve_timestamp = timestamp()
        host.section = self.name
        self.solved_hosts[host.inv] = host

    def merge_hosts(self, previous_hosts):
        """Current implementation: discard any new hosts (that are missing from previous_hosts),
        add all missing previous hosts (that exist in previous_hosts but missing from current).
        Future implementation: add flag to keep new hosts.
        :param previous_hosts - dict of inv -> host data from previous run
        :type previous_hosts: dict
        """

        new_hosts = self.problem_hosts.keys() - previous_hosts.keys()
        for inv in new_hosts:
            del self.problem_hosts[inv]

        problem_hosts = previous_hosts.keys() & self.problem_hosts.keys()
        for inv in problem_hosts:
            self.update_problem_host(previous_hosts[inv])

        section_name = self.name
        solved_hosts = previous_hosts.keys() - self.problem_hosts.keys()

        for inv in solved_hosts:
            host = previous_hosts[inv]
            if host.section == section_name:
                self.add_solved_host(host)

    def empty(self):
        """Return True if section does not contain any hosts."""
        return not self.problem_hosts and not self.solved_hosts

    def has_problem_hosts(self):
        return bool(self.problem_hosts)

    def text(self):
        return self.formatter.format_section(self)


class ReportFormatter:
    def format(self, sections):
        report = "\n\n".join(section.text() for section in sections if not section.empty())
        return report


class SectionFormatter:
    def __init__(self, title):
        self.title = title

        self.stand_url_ui = config.get_value("stand.ui_url").rstrip("/")
        self._stand_name = config.get_value("stand.name", default=self.stand_url_ui)

    def format_section(self, section):
        return self.format_title() + "\n\n" + self.format_section_hosts(section)

    def format_section_hosts(self, section):
        """

        :type section: ReportSection
        """
        output = []
        all_hosts = itertools.chain(section.problem_hosts.values(), section.solved_hosts.values())
        all_hosts = sorted(all_hosts, key=lambda h: (h.project, h.name, h.solved))

        for host in all_hosts:
            line = self.wiki_format_host(host.host_uuid, host.name, host.status)
            if host.reason:
                line += self.wiki_format_reason(host.reason, solved=host.solved)
            output.append(line)

        return "\n".join(output)

    def format_title(self):
        return "====Wall-E{stand} {title}:".format(stand=self.padded_stand_name, title=self.title)

    def wiki_format_host(self, uuid, hostname, status):
        return "  * **(({base}/host/{uuid} {name})) [{status}]**: ".format(
            base=self.stand_url_ui, uuid=uuid, name=hostname, status=status
        )

    @staticmethod
    def host_failure_reason(reason_str):
        return '""{}""'.format(reason_str)

    @staticmethod
    def resolved_failure_reason(formatted_reason):
        return "--{}--".format(formatted_reason)

    @property
    def padded_stand_name(self):
        """Return padded stand name useful for title formatting. Empty string for production."""
        production_stand_name = "Production"
        if self._stand_name and self._stand_name != production_stand_name:
            return " " + self._stand_name
        else:
            return ""

    @classmethod
    def wiki_format_reason(cls, reason, solved=False):
        if solved:
            return cls.resolved_failure_reason(reason)
        else:
            return reason


class GroupingSectionFormatter(SectionFormatter):
    """Format section into wiki-formatting. Produce section title and group hosts by project."""

    def format_section_hosts(self, section):
        """

        :type section: ReportSection
        """
        output = []
        all_hosts = itertools.chain(section.problem_hosts.values(), section.solved_hosts.values())
        all_hosts = sorted(all_hosts, key=lambda h: (h.project, h.name, h.solved))

        for project, hosts in itertools.groupby(all_hosts, attrgetter("project")):
            output.append(self.subtitle(project))

            for host in hosts:
                line = self.wiki_format_host(host.host_uuid, host.name, host.status)
                if host.reason:
                    line += self.wiki_format_reason(host.reason, solved=host.solved)
                output.append(line)

        return "\n".join(output)

    def subtitle(self, project):
        return "=====(({base}/project/{project} {project}))".format(base=self.stand_url_ui, project=project)
