import datetime
import logging
import re

from walle.constants import CMS_REJECT_RE
from walle.failure_reports import observers
from walle.failure_reports.base import ErrorHostsReport, HostFailureRotationStrategy, ReportFormatter, StreamKey
from walle.failure_reports.reports import HostsByStatusCollector, ReportContents
from walle.failure_reports.startrek import StarTrekReportPublisher, TicketParams
from walle.hosts import HostStatus
from walle.projects import Project
from walle.util.limits import parse_period

log = logging.getLogger(__name__)


CMS_TICKET_TAGS = ['wall-e', 'cms', 'cms_error']


class CmsReportParams:
    _DEFAULT_ACQUIRE_TIMEOUT = "1w"

    def __init__(
        self, url_match=None, queue=None, responsible=None, followers=None, acquire_timeout=None, dry_run=None
    ):
        self.url_match = url_match
        self.queue = queue
        self.responsible = responsible
        self.followers = followers
        self.acquire_timeout = parse_period(acquire_timeout or self._DEFAULT_ACQUIRE_TIMEOUT)
        self.dry_run = dry_run


class CmsReport:
    def __init__(self, cms_report_params):
        self.report_params = cms_report_params
        self.projects = list(Project.objects(cms=re.compile(self.report_params.url_match)))
        self._ticket_params = None

    def ticket_params(self):
        if not self._ticket_params:
            self._ticket_params = TicketParams(
                tags=CMS_TICKET_TAGS,
                queue=self.report_params.queue,
                assignee=self.report_params.responsible,
                followers=self.report_params.followers,
                unique_salt=self._unique_salt(),
            )

        return self._ticket_params

    def publish(self):
        log.info("publishing CMS-report with re: %s", self.report_params.url_match)
        log.debug("cms_url: '%s' projects: %s", self.report_params.url_match, ",".join(str(p) for p in self.projects))
        with self._create_report() as report:
            report_data = CMSDeniedHosts()
            report_contents = ReportContents(_CmsReportFormatter())
            report_contents.add_section(report_data.gather_data(self.projects))
            report.set_content("Wall-e CMS errors", report_contents)

    def _create_report(self):
        publisher = StarTrekReportPublisher(self.ticket_params(), dry_run=self.report_params.dry_run)
        rotation_strategy = HostFailureRotationStrategy(rotate=False)
        stream_key = StreamKey(publisher.get_stream_key(), publisher.name, rotation_strategy.name)
        observers_group = observers.collect_group(stream_key)

        return ErrorHostsReport(
            stream_key.wrapped_key(),
            publisher,
            rotation_strategy,
            observers_group,
            raise_on_failure=False,
            dry_run=self.report_params.dry_run,
        )

    @staticmethod
    def _unique_salt():
        # TODO: make unique salt more reliable
        # using hour-based value as we assume hourly report cycle
        return datetime.datetime.now().strftime("%Y-%m-%d-%H")


def create_reports(report_params, dry_run):
    return [CmsReport(CmsReportParams(dry_run=dry_run, **cms_rp)) for cms_rp in report_params]


class _CmsReportFormatter(ReportFormatter):
    REPORT_FOOTER = (
        "\n\n----\n"
        "Назначение и содержание этого тикета описано "
        "((https://docs.yandex-team.ru/wall-e/guide/reports#cms-reports в документации по Wall-E)), "
        "там же описаны доступные настройки для таких отчётов. "
        "Пояснения ((https://docs.yandex-team.ru/wall-e/faq#faq-reports "
        "в часто задаваемых вопросах)) тоже могут быть полезны."
    )

    def format(self, sections):
        return super().format(sections) + self.REPORT_FOOTER


class CMSDeniedHosts(HostsByStatusCollector):
    """Report hosts that failed due to CMS decision"""

    request_fields = HostsByStatusCollector.request_fields + ("status_reason",)
    hosts_query = {"status": HostStatus.DEAD, "status_reason": re.compile(CMS_REJECT_RE)}
    section_title = "has cms-dead hosts"
