"""Module collects routines that report broken hosts to startrek to their owners."""

import itertools
import logging
from copy import deepcopy

from sepelib.core import config
from walle.failure_reports import reports, observers, cms_reports
from walle.failure_reports.base import (
    ErrorHostsReport,
    relative_today,
    StreamKey,
    DailyReportRotationStrategy,
    ReportFormatter,
)
from walle.failure_reports.project_reports import ticket_params_for_all_projects
from walle.failure_reports.startrek import StarTrekReportPublisher
from walle.models import timestamp
from walle.util.workdays import (
    next_working_hour_timestamp,
    WORKING_DAYS,
    REPORTS_WORKING_HOURS,
    WEEKDAY_MON,
    from_timestamp,
)

log = logging.getLogger(__name__)


def _check_hosts(
    projects,
    reports_cms_projects,
    unreachable_hosts,
    top_problematic_hosts,
    infrastructure_problems,
    maintenance_ticket_stuck,
    broken_dns_hosts,
):
    report_content = reports.ReportContents(_DailyReportFormatter())

    dead_hosts_report = reports.DeadHosts(reports_cms_projects)
    report_content.add_section(dead_hosts_report.gather_data(projects))

    invalid_hosts_report = reports.InvalidHosts()
    report_content.add_section(invalid_hosts_report.gather_data(projects))

    error_hosts_report = reports.ErrorHosts()
    report_content.add_section(error_hosts_report.gather_data(projects))

    if unreachable_hosts:
        unreachable_hosts_report = reports.UnreachableHosts()
        report_content.add_section(unreachable_hosts_report.gather_data(projects))

    if broken_dns_hosts:
        broken_dns_report = reports.BrokenDnsHosts()
        report_content.add_section(broken_dns_report.gather_data(projects))

    if infrastructure_problems:
        infrastructure_problems_report = reports.InfrastructureProblems()
        report_content.add_section(infrastructure_problems_report.gather_data(projects))

    if top_problematic_hosts:
        top_failing_report = reports.TopProblematicHosts()
        report_content.add_section(top_failing_report.gather_data(projects))

    if maintenance_ticket_stuck:
        maintenance_too_long = reports.OldMaintenanceTicketHosts()
        maintenance_too_long.gather_data(projects)
        report_content.add_section(maintenance_too_long.gather_data(projects))

    return report_content


def host_failure_reporter():
    now = timestamp()
    # First-shot simplest implementation:
    # * Saturday and Sunday are holidays
    # * 23:00 pm till 09:00 am is a curfew (e.g, a quiet hour)
    if now != next_working_hour_timestamp(now, WORKING_DAYS, *REPORTS_WORKING_HOURS):
        return

    report_params = deepcopy(config.get_value("failure_reports.report_params"))
    dry_run = config.get_value("failure_reports.dry_run", True)

    # Add top failing hosts every monday, but only if enabled.
    # No per-project customization yet.
    report_params["top_problematic_hosts"] &= from_timestamp(now).isoweekday() == WEEKDAY_MON

    # reports about hosts stuck in CMS
    cms_report_params = report_params.pop("cms_report_params")
    reports_cms = cms_reports.create_reports(cms_report_params, dry_run)
    reports_cms_projects = list(itertools.chain.from_iterable([report.projects for report in reports_cms]))

    log.info("%s CMS reports configured, total projects: %s", len(reports_cms), len(reports_cms_projects))

    for report in reports_cms:
        try:
            report.publish()
        except Exception as error:
            log.exception(error)

    for ticket_params, projects in ticket_params_for_all_projects():
        with _create_report(ticket_params, dry_run) as report:
            summary = _get_summary(ticket_params.get("summary", None))
            error_reports = _check_hosts(projects, reports_cms_projects, **report_params)

            report.set_content(summary, error_reports)


def _create_report(ticket_params, dry_run):
    publisher = StarTrekReportPublisher(ticket_params, dry_run=dry_run)
    rotation_strategy = DailyReportRotationStrategy()
    stream_key = StreamKey(publisher.get_stream_key(), publisher.name, rotation_strategy.name)
    observers_group = observers.collect_group(stream_key)

    return ErrorHostsReport(
        stream_key.wrapped_key(), publisher, rotation_strategy, observers_group, raise_on_failure=False, dry_run=dry_run
    )


def _get_summary(summary):
    return _get_summary_for_date(_get_summary_base(summary), relative_today())


def _get_summary_base(summary=None):
    summary_parts = filter(None, ("Error hosts in Wall-E", summary))
    summary = " | ".join(summary_parts)

    return summary


def _get_summary_for_date(summary_base, date):
    summary = "{} ({})".format(summary_base, date.strftime("%d.%m.%Y"))
    return summary


class _DailyReportFormatter(ReportFormatter):
    def format(self, sections):
        return (
            super().format(sections) + "\n\n----\n"
            "Назначение и содержание этого тикета описано "
            "((https://docs.yandex-team.ru/wall-e/guide/reports в документации по Wall-E)), "
            "там же описаны доступные настройки для таких отчётов. "
            "Пояснения ((https://docs.yandex-team.ru/wall-e/faq#faq-reports "
            "в часто задаваемых вопросах)) тоже могут быть полезны."
        )
