"""Some additional report processing."""
import logging
from contextlib import contextmanager

from mongoengine import Q

from walle import audit_log
from walle.authorization import ISSUER_WALLE
from walle.failure_reports.base import ReportObserver, ErrorReportModel, now
from walle.hosts import Host
from walle.util import notifications
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)


def collect_group(stream_key):
    return ObserverGroup(
        [HostTicketForReport(), AuditLogForReport(), NotifyProjectOwners(), StoreReport(stream_key.wrapped_key())]
    )


@contextmanager
def _catch_and_log_event_error(event_name):
    try:
        yield
    except Exception:
        log.exception("Failed to pass '{}' event to observer".format(event_name))


class ObserverGroup(ReportObserver):
    def __init__(self, observers):
        """
        :type observers: Iterable[ReportObserver]
        """
        self._observers = tuple(observers)

    def map(self, event_name, method):
        for observer in self._observers:
            with _catch_and_log_event_error(event_name):
                method(observer)

    def report_created(self, report_key, report_hosts, previous_report_key=None):
        self.map(
            "report created", lambda observer: observer.report_created(report_key, report_hosts, previous_report_key)
        )

    def report_failed_to_create(self, error_message, report_hosts):
        self.map(
            "report failed to create", lambda observer: observer.report_failed_to_create(error_message, report_hosts)
        )

    def report_updated(self, report_key, report_hosts):
        self.map("report updated", lambda observer: observer.report_updated(report_key, report_hosts))

    def report_closed(self, report_key, report_hosts):
        self.map("report closed", lambda observer: observer.report_closed(report_key, report_hosts))


class HostTicketForReport(ReportObserver):
    def report_created(self, report_key, report_hosts, previous_report_key=None):
        """Set current report's key as a host's ticket for hosts which do not have tickets yet."""
        self._set_hosts_tickets(report_key, report_hosts, previous_report_key)

    def report_updated(self, report_key, report_hosts):
        """Clear host's tickets for hosts that were removed from the report, set ticket for hosts that were added."""
        self._set_hosts_tickets(report_key, report_hosts)
        self._unset_solved_hosts_tickets(report_key, report_hosts)

    def report_closed(self, report_key, report_hosts):
        """Clear host's tickets from hosts when report/ticket gets closed."""
        self._unset_closed_report_ticket(report_key, report_hosts)

    @staticmethod
    def _set_hosts_tickets(report_key, report_hosts, previous_report_key=None):
        inv_list = [host.inv for host in report_hosts if not host.solved]
        if not inv_list:
            return

        ticket_clause = Q(ticket__exists=False)
        if previous_report_key is not None:
            ticket_clause |= Q(ticket=previous_report_key)

        for host in gevent_idle_iter(Host.objects(ticket_clause, inv__in=inv_list)):
            host.update(set__ticket=report_key)

    @staticmethod
    def _unset_solved_hosts_tickets(report_key, report_hosts):
        inv_list = {host.inv for host in report_hosts if host.solved}
        if not inv_list:
            return

        for host in gevent_idle_iter(Host.objects(inv__in=inv_list, ticket=report_key)):
            host.update(unset__ticket=True)

    @staticmethod
    def _unset_closed_report_ticket(report_key, report_hosts):
        inv_list = [host.inv for host in report_hosts]
        if not inv_list:
            return

        for host in gevent_idle_iter(Host.objects(ticket=report_key, inv__in=inv_list)):
            host.update(unset__ticket=True)


class AuditLogForReport(ReportObserver):
    def report_created(self, report_key, report_hosts, previous_report_key=None):
        for host in report_hosts:
            if not host.solved and not host.audit_log_id:
                self._open_audit_log_for_report_host(host, report_key)

    def report_failed_to_create(self, error_message, report_hosts):
        for host in report_hosts:
            if not host.solved and not host.audit_log_id:
                log_entry = self._open_audit_log_for_report_host(host, report_key=None)
                audit_log.fail_request(log_entry, error_message)

    def report_updated(self, report_key, report_hosts):
        for host in report_hosts:
            if host.solved and host.audit_log_id:
                self._close_audit_log_for_host(host)

            if not host.solved and not host.audit_log_id:
                return self._open_audit_log_for_report_host(host, report_key)

    def report_closed(self, report_key, report_hosts):
        for host in report_hosts:
            if host.audit_log_id:
                self._close_audit_log_for_host(host)

    @staticmethod
    def _open_audit_log_for_report_host(report_host, report_key=None):
        audit_log_params = {
            "issuer": ISSUER_WALLE,
            "project_id": report_host.project,
            "inv": report_host.inv,
            "name": report_host.name,
            "host_uuid": report_host.host_uuid,
            "ticket": report_key,
            "reason": report_host.reason,
        }

        with audit_log.on_report(**audit_log_params) as audit_log_entry:
            report_host.audit_log_id = audit_log_entry.id

        return audit_log_entry

    @staticmethod
    def _close_audit_log_for_host(report_host):
        if report_host.solved:
            audit_log.complete_request(report_host.audit_log_id, check_updated=False)
        else:
            audit_log.cancel_request(report_host.audit_log_id)

        del report_host.audit_log_id


class NotifyProjectOwners(ReportObserver):
    def report_failed_to_create(self, error_message, report_hosts):
        projects = {host.project for host in report_hosts}
        for project_id in projects:
            notifications.on_failed_to_create_report(project_id, error_message)


class StoreReport(ReportObserver):
    def __init__(self, stream_key):
        self._stream_key = stream_key

    def report_created(self, report_key, report_hosts, previous_report_key=None):
        ErrorReportModel.create(
            report_key=report_key, stream_key=self._stream_key, report_date=now(), hosts=report_hosts
        )

    def report_updated(self, report_key, report_hosts):
        ErrorReportModel.update(report_key, hosts=report_hosts)

    def report_closed(self, report_key, report_hosts):
        ErrorReportModel.update(report_key, hosts=report_hosts, closed=True)
