import logging
from collections import defaultdict

from sepelib.core import config
from sepelib.core.exceptions import Error
from walle.clients import staff, bot
from walle.failure_reports.startrek import TicketParams
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)


class NoAssigneeException(Error):
    def __init__(self, projects):
        message = "Can not assign report to anybody, projects '{}' have no owners."
        projects_ids = ", ".join([p.id for p in projects])
        super().__init__(message, projects_ids)


def ticket_params_for_all_projects():
    """Collect projects grouped by task parameters, one task per group will be created."""
    return _reports_for_projects(_fetch_projects())


def ticket_params_for_project(project):
    """Try to collect report params for project. Return None if it is not possible for some obscure reason."""

    default_report = config.get_value("failure_reports.issue_params")
    project_groups = _group_projects([project], default_report)

    for ticket_params, projects in project_groups.items():
        return _ensure_assignee(ticket_params, projects, default_report["queue"])

    return None


def _reports_for_projects(projects):
    """Group given projects by task parameters, one task per group will be created."""

    default_report = config.get_value("failure_reports.issue_params")
    project_groups = _group_projects(projects, default_report)

    for ticket_params, projects in project_groups.items():
        try:
            ticket_params = _ensure_assignee(ticket_params, projects, default_report["queue"])
        except NoAssigneeException as e:
            # no owners for project, skip it
            log.warn(str(e))
            continue

        yield ticket_params, projects


def _ensure_assignee(ticket_params, projects, default_queue):
    if "assignee" in ticket_params:
        return ticket_params

    if ticket_params["queue"] != default_queue:
        # project have it's own queue, they can do whatever they want there
        return ticket_params

    owners = _get_all_owners(projects)

    if not owners:
        raise NoAssigneeException(projects)

    assignee = owners[0]
    followers = ticket_params.get("followers", ())
    followers = list(set(tuple(owners[1:]) + followers)) or None

    return ticket_params.append(assignee=assignee, followers=followers)


def _get_all_owners(projects):
    owners = {owner for p in projects for owner in p.owners}
    return staff.resolve_owners(tuple(sorted(owners)), allow_robots=False)


def _fetch_projects():
    return Project.objects(reports__enabled__ne=False).only("id", "name", "owners", "bot_project_id", "reports")


def _group_projects(projects, default_report):
    groups = defaultdict(list)
    default_report = TicketParams.from_ticket_params(default_report)
    bot_projects = bot.get_oebs_projects()

    for project in gevent_idle_iter(projects):
        if project.reports:
            parameters = project.reports
            if not parameters.enabled:
                continue

            ticket_params = dict(parameters.extra or {})
            ticket_params["queue"] = parameters.queue
            if parameters.summary:
                ticket_params["summary"] = parameters.summary

            # Create one ticket for all projects with same ticket parameters
            report_params = TicketParams.from_ticket_params(ticket_params)
        elif project.bot_project_id:
            # Group tickets by bot project id (effectively, ABC service id)
            summary = bot_projects[project.bot_project_id]["en_description"]
            report_params = default_report.append(summary=summary)
        else:
            # Try to group projects by first word in their id fields.
            # This will work for some slowpoke-projects which do not have custom settings,
            # but have their projects split up into many projects for any reasons.
            # Example: lbk-man, lbk-sas, lbk-mtn, etc, or metrika, metrika-mtn.
            # Although, this may produce some corner cases, like search-dev or search-pumpkin.
            # This can be resolved by assigning (different) bot project ids to those projects.

            # N.B. I don't really know, may be we do not need it at all.
            summary = project.id.partition("-")[0]
            report_params = default_report.append(summary=summary)

        groups[report_params].append(project)

    return groups
