import re
import textwrap
import datetime
import time
from copy import deepcopy

from sepelib.core import config
from walle.failure_reports.base import ReportFormatter
from walle.failure_reports.startrek import TicketParams
from walle.fsm_stages import startrek_report
from walle.util.misc import get_location_path

TEXT_TEMPLATE = """
Hello, YandexDataCenterTeam!

All nodes in rack {queue}/{rack} {error}. {short_solution}

Comment:
Grafana: ((https://grafana.yandex-team.ru/d/f_EZJFaiz/itdc-alldc-rack-detail?orgId=1&refresh=1m&var-CITY={city}&var-SITE={datacenter}&var-MODULE={grafana_queue}&var-RACK={rack}&from=now-2d&to=now
<{{List of nodes
{nodes}
}}>
Error: (({error_link} Rack_{error}))

Thank you.
"""

TICKET_RACK_REPAIR_INTERVAL_HOUR = 3 * 60 * 60


class UnknownHostLocation(startrek_report.CanNotCreateReports):
    def __init__(self, location):
        location_path = get_location_path(location, "queue")
        super().__init__("Can not find host location in location map: {}", location_path)


def get_itdc_location(location):
    location_map = {
        ("RU", "MOW", "UGRB"): "UGR",
        ("RU", "SAS", "SASTA"): "SAS",
        ("RU", "MYT", "MPLMASS"): "MYT",
        ("FI", "MANTSALA", "A"): "MAN",
        ("FI", "MANTSALA", "B"): "MAN",
        ("RU", "IVA", "IVNIT"): "IVA",
        ("RU", "MOW", "FOL"): "FOL",
        ("NL", "AMS", "AM2"): "AMS",
        ("US", "ASHBURN", "DC4"): "ASH",
        ("RU", "VLADIMIR", "ALPHA"): "VLA",
        ("RU", "VLADIMIR", "BETA"): "VLA",
    }

    try:
        return location_map[(location.country, location.city, location.datacenter)]
    except KeyError:
        raise UnknownHostLocation(location)


def _grafana_queue_name(queue):
    # this is gonna need some maintenance, I am sure there gonna be more weird rules :-(
    return re.sub(r"[\W_]", "-", queue)


def _get_uniq_ticket_salt():
    rounded_now = int(time.time())
    rounded_now = rounded_now - rounded_now % TICKET_RACK_REPAIR_INTERVAL_HOUR
    return datetime.datetime.fromtimestamp(rounded_now).strftime("%Y-%m-%d-%H")


def get_report(location, error, error_short, error_link, solution_short):
    itdc_location = get_itdc_location(location)

    defaults = deepcopy(config.get_value("failure_reports.rack_{}".format(error_short)))
    defaults["components"] = [itdc_location]
    defaults["tags"] = sorted([itdc_location, location.queue, "rack-{}".format(location.rack)])
    defaults["unique_salt"] = _get_uniq_ticket_salt()

    ticket_params = TicketParams.from_ticket_params(defaults)

    report_summary = "Rack {error_short}: {queue} rack {rack}".format(
        error_short=error_short, queue=location.queue, rack=location.rack
    )

    return startrek_report.get_report(
        ticket_params, report_summary, RackReportFormatter(location, error, error_link, solution_short)
    )


class RackReportFormatter(ReportFormatter):
    def __init__(self, location, error, error_link, short_solution):
        self.location = location
        self.error = error
        self.error_link = error_link
        self.short_solution = short_solution

    def format(self, sections):
        return self.template().format(
            nodes=super().format(sections),
            city=self.location.city,
            datacenter=self.location.datacenter,
            error=self.error,
            error_link=self.error_link,
            short_solution=self.short_solution,
            queue=self.location.queue,
            grafana_queue=_grafana_queue_name(self.location.queue),
            rack=self.location.rack,
        )

    @staticmethod
    def template():
        return textwrap.dedent(TEXT_TEMPLATE)
