"""This module represents startrek report."""

import logging
from collections import OrderedDict
from collections.abc import Mapping

import six

from sepelib.yandex.startrek import Relationship
from walle.clients import startrek
from walle.clients.startrek import StartrekClientError, StartrekClientRequestError, StartrekClientConnectionError
from walle.clients.utils import retry
from walle.failure_reports.base import ReportPublisher, ReportPublisherFailure
from walle.util.misc import drop_none, first

log = logging.getLogger(__name__)


class StarTrekReportPublisher(ReportPublisher):
    """Manipulate a StarTrek ticket for error hosts.
    Can create a new ticket if needed, and close previous tickets if any.
    """

    name = "startrek"

    _startrek = None
    _ticket_params = None
    _dry_run = True

    def __init__(self, ticket_params, dry_run=False):
        """

        :type ticket_params: TicketParams
        """
        self._startrek = startrek.get_client()
        self._ticket_params = ticket_params
        self._dry_run = dry_run

        super().__init__()

    def get_stream_key(self):
        return self._ticket_params.stream_key()

    @classmethod
    def from_stream_key(cls, stream_key):
        return cls(TicketParams.from_stream_key(stream_key))

    def close_old_report(self, report_key):
        try:
            if not self._dry_run:
                self._startrek.close_issue(report_key, **self._ticket_params.closing_kwargs())
        except StartrekClientRequestError as e:
            if e.response.status_code == 404:
                # Either ticket does not exist or it is closed already.
                # Either way we are good to ignore it.
                result = e.response.json()
                log.warning("Failed to close ticket %s: %s", report_key, ", ".join(result["errorMessages"]))
                return

            log.error("Failed to close ticket %s: %s", report_key, str(e))
            raise ReportPublisherFailure("Failed to close startrek ticket {}: {}", report_key, str(e))

        except StartrekClientError as e:
            log.error("Failed to close ticket %s: %s", report_key, str(e))
            raise ReportPublisherFailure("Failed to close startrek ticket {}: {}", report_key, str(e))

        except Exception:
            log.exception("Failed to close ticket %s", report_key)
            raise ReportPublisherFailure("Failed to close startrek ticket {}", report_key)

        else:
            log.info("Closed ticket %s (dry_run=%s)", report_key, self._dry_run)

    def create_new_report(self, summary, report_text, report_hosts, previous_report=None):
        new_report_key = None  # for dry run mode.

        if not self._dry_run:
            unique_field = self._ticket_params.unique_field(summary)
            ticket = self._check_ticket_by_unique(unique_field)

            if ticket is None:
                ticket = self._create_ticket(
                    summary,
                    report_text,
                    unique_field=unique_field,
                    linked_tickets=self._linked_tickets(report_hosts, previous_report),
                )

            new_report_key = ticket["key"]

        log.info("Created new ticket '%s' (%s) (dry_run=%s)", summary, new_report_key, self._dry_run)
        return new_report_key

    def update_existing_report(self, report, report_text, report_hosts):
        try:
            if not self._dry_run:
                self._startrek.modify_issue(
                    report.report_key,
                    drop_none({"description": report_text, "links": self._linked_tickets(report_hosts) or None}),
                )
        except StartrekClientError as e:
            log.error("Failed to update ticket %s: %s", report.report_key, str(e))
            raise ReportPublisherFailure("Failed to update startrek ticket {}: {}", report.report_key, str(e))

        except Exception:
            log.exception("Failed to update ticket %s", report.report_key)
            raise ReportPublisherFailure("Failed to update startrek ticket {}", report.report_key)

        else:
            log.info("Updated ticket %s (dry_run=%s)", report.report_key, self._dry_run)

    def verify_report_published(self, report_key):
        try:
            ticket = self._startrek.get_issue(report_key)
        except StartrekClientRequestError as e:
            if e.response.status_code == 404:  # ticket does not exist
                log.warn("Startrek ticket %s does not exist.", report_key)
                return False
            else:
                log.error("Failed to get status for startrek ticket %s: %s.", report_key, str(e))
                raise ReportPublisherFailure("Failed to get status for startrek ticket {}: {}", report_key, str(e))

        except Exception:
            log.exception("Failed to get status for startrek ticket %s.", report_key)
            raise ReportPublisherFailure("Failed to get status for startrek ticket {}", report_key)

        if "resolution" in ticket:
            log.debug("Startrek ticket %s is closed: %s.", report_key, ticket["resolution"]["key"])
            return False

        return True

    @staticmethod
    def _linked_tickets(report_hosts, previous_report=None):
        links = OrderedDict()  # maintain order for tests
        new_report_hosts = set()

        for host in report_hosts:
            for ticket in host.tickets or ():
                links[ticket] = {"relationship": Relationship.RELATES, "issue": ticket}
            new_report_hosts.add(host.inv)

        if previous_report is not None:
            links[previous_report.report_key] = {
                "relationship": Relationship.RELATES,
                "issue": previous_report.report_key,
            }

            for host in previous_report.hosts:
                if host.inv in new_report_hosts:
                    for ticket in host.tickets or []:
                        links[ticket] = {"relationship": Relationship.RELATES, "issue": ticket}

        return list(links.values())

    def _create_ticket(self, summary, report_text, linked_tickets, unique_field):
        try:
            ticket = self._create_ticket_attempt(linked_tickets, report_text, summary, unique_field)
        except StartrekClientConnectionError as e:
            log.error("Failed to create ticket: %s", str(e))
            raise ReportPublisherFailure("Failed to create startrek ticket: {}", str(e))

        except StartrekClientRequestError as e:
            ticket = self._check_ticket_by_unique(unique_field)

            if ticket:
                log.info("Found existing ticket by unique field: %s", unique_field)
            else:
                log.error("Error creating startrek ticket: %s", str(e))
                raise ReportPublisherFailure("Failed to create startrek ticket: {}", str(e))

        except Exception:
            log.exception("Failed to create ticket")
            raise ReportPublisherFailure("Failed to create startrek ticket")

        return ticket

    @retry(interval=1, backoff=2, exceptions=(StartrekClientConnectionError,))
    def _create_ticket_attempt(self, linked_tickets, report_text, summary, unique_field):
        ticket = self._startrek.create_issue(
            dict(
                self._ticket_params.append(
                    summary=summary,
                    description=report_text,
                    links=linked_tickets,
                    unique=unique_field,
                )
            )
        )
        return ticket

    def _check_ticket_by_unique(self, unique_field):
        """return the key of the first of found tickets (expect to find only one ticket) or None"""

        unique_field = six.ensure_str(unique_field, "utf-8")

        try:
            tickets = self._startrek.get_issues(filter={"unique": unique_field})
        except StartrekClientError:
            return None  # raise original error from "create ticket"

        return first(iter(tickets))


class TicketParams(Mapping):
    """Immutable collection of ticket params suitable for using as a dict key when collecting projects into groups."""

    def __init__(self, **params):
        # cleaned params make up a stream key
        self._clean_params = self._cleanup_params(params)

        # these params used for some other purpose
        self._unique_salt = params.pop("unique_salt", "")
        self._closing_transition = params.pop("close_transition", None)

        # these are ticket params for startrek
        self._params = params

    def to_dict(self):
        dct_params = {"unique_salt": self._unique_salt, "close_transition": self._closing_transition}
        dct_params.update(self._params)
        return drop_none(dct_params)

    @classmethod
    def from_dict(cls, dct):
        dct = {key: cls._pound_sign_to_underscore(value) for key, value in dct.items()}
        return cls(**dct)

    @classmethod
    def from_ticket_params(cls, params):
        return cls._normalized(params)

    @classmethod
    def from_stream_key(cls, stream_key):
        ticket_params = dict(token.split(":") for token in stream_key.split("#"))
        return cls._normalized(ticket_params)

    def stream_key(self):
        return "#".join('{}:{}'.format(k, self._string(v)) for k, v in self._clean_params)

    def unique_field(self, summary):
        return "{}#{}#{}".format(self.stream_key(), self._unique_salt, summary)

    def append(self, **params):
        """Return copy of current params with specified params appended and probably replaced."""
        return self.from_ticket_params(dict(self, **params))

    def closing_kwargs(self):
        return drop_none({"transition": self._closing_transition})

    def __hash__(self):
        return hash(self.stream_key())

    def __eq__(self, other):
        if type(self) == type(other):
            # use only "important" parameters - to group projects into groups
            return self._clean_params == other._clean_params
        else:
            return False

    def __str__(self):
        return "{}(**{})".format(type(self).__name__, dict(self))

    __repr__ = __str__

    @classmethod
    def _cleanup_params(cls, params):
        exclude = {"followers", "links", "unique", "unique_salt"}
        return sorted((k, cls._pound_sign_to_underscore(v)) for k, v in params.items() if k not in exclude)

    @classmethod
    def _normalized(cls, ticket_params):
        exclude = {"unique", "unique_salt"}
        normalized_params = dict()

        for key, value in ticket_params.items():
            if key not in exclude:
                value = cls._pound_sign_to_underscore(value)

            if value is None:
                continue

            if key.lower() in {"components", "tags", "followers"}:
                value = cls._string_list(value)
                value = [cls._name_or_id(val) for val in value]

            elif key.lower() == "project":
                value = cls._name_or_id(value)

            normalized_params[key] = value

        return cls(**normalized_params)

    @staticmethod
    def _name_or_id(val):
        if isinstance(val, str):
            return int(val) if val.isdigit() else val.replace("#", "_")
        else:
            return val

    @staticmethod
    def _string_list(value):
        if isinstance(value, (list, tuple, set)):
            return sorted(value)

        return sorted(str(value).split(","))

    @staticmethod
    def _string(value):
        if isinstance(value, (list, tuple, set)):
            return ",".join(sorted(map(str, value)))

        return str(value)

    @classmethod
    def _pound_sign_to_underscore(cls, value):
        if isinstance(value, (list, tuple, set)):
            return [cls._pound_sign_to_underscore(val) for val in value]

        if isinstance(value, str):
            return value.replace("#", "_")

        return value

    def __iter__(self):
        return iter(self._params)

    def __getitem__(self, key):
        return self._params[key]

    def __len__(self):
        return len(self._params)
