"""Pushes host FQDN to CAuth to speed up process of allowing users' access to it"""

import logging

from sepelib.core import config
from sepelib.core.constants import MINUTE_SECONDS
from walle.clients import cauth
from walle.fsm_stages.common import register_stage, complete_current_stage, get_current_stage, fail_current_stage
from walle.stages import Stages

log = logging.getLogger(__name__)

# This stage is just a faster alternative to cauth pull, which occurs every 15 minutes
# So we can retry push for 10 minutes, then it loses any sense
STAGE_TIMEOUT = 10 * MINUTE_SECONDS


def _add_host_to_cauth_via_push(host):
    if config.get_value("cauth.add_hosts_via_push", False):
        try:
            _push_host(host)
        except cauth.CAuthError as e:
            return _retry_or_fail_stage_on_exc(host, e)
        except Exception as e:
            _log_exception(host, ", {}".format(e))
            fail_current_stage(host, "Push host to CAuth failed: {}".format(e))
            return
        else:
            log.info("Successfully pushed host %s with project %s into CAuth", host.name, host.project)
    else:
        log.info("Push to CAuth is disabled, skipping")

    complete_current_stage(host)


def _push_host(host):
    project = host.get_project()
    cauth_params = project.cauth_settings.to_request_params() if project.cauth_settings else {}

    log.info("Pushing host %s into CAuth with project %s and params: %s", host.name, host.project, cauth_params)
    cauth.add_host_via_push(host.name, host.project, cauth_params)


def _retry_or_fail_stage_on_exc(host, exc):
    if get_current_stage(host).timed_out(STAGE_TIMEOUT):
        _log_exception(host, ", {}".format(exc))
        fail_current_stage(host, "Push host to CAuth stage timed out. {}".format(exc))
    else:
        raise exc  # FSM will retry the stage


def _log_exception(host, action):
    log.exception(
        "%s: Failed to add hostname in project %s into CAuth via push{}".format(action), host.human_name(), host.project
    )


register_stage(Stages.ADD_HOST_TO_CAUTH, _add_host_to_cauth_via_push)
