"""Contains all logic for allocating name for the host."""

import logging
from functools import partial
from itertools import chain, count

import mongoengine

import walle.tasks
from sepelib.core import constants
from sepelib.core.exceptions import Error, LogicalError
from walle import network
from walle.clients import bot
from walle.errors import InvalidHostConfiguration
from walle.fsm_stages.common import (
    NEXT_CHECK_ADVANCE_TIME,
    register_stage,
    get_current_stage,
    commit_stage_changes,
    complete_current_stage,
    fail_current_stage,
)
from walle.hosts import Host
from walle.models import timestamp
from walle.network import BlockedHostName
from walle.stages import Stages
from walle.util.cache import cached_with_exceptions
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


def _allocate_hostname(host):
    # TODO: Attention: There is a very troublesome problem here which we don't know how to fix right now:
    # We change the host name in our database here. If everything goes fine and after this we'll rename the host in
    # BOT - we're cool and everything will work as expected. But if user cancels the task before we rename the host in
    # BOT or if we fail to rename it in BOT, we are in trouble: the host has one name in our database and different name
    # in BOT, so after some time Wall-E's GC will invalidate it when it will sync database with BOT. We have no idea how
    # to fix this right now. Perhaps we need task cancellation mechanism.

    stage = get_current_stage(host)

    if stage.get_param("free", False):
        template = network.get_free_host_name_template()
        res = _check_if_host_name_already_ok(host, template)
        if res is not None:
            return
        _try_assign_hostname(host, template.fill(host.inv))
        complete_current_stage(host)
        return
    else:
        try:
            template = network.get_host_name_template(host)
        except InvalidHostConfiguration as e:
            fail_current_stage(host, "Failed to create template for host name generation: {}".format(e))
            return

        res = _check_if_host_name_already_ok(host, template)
        if res is not None:
            return res

        filter_matching = partial(filter, template.shortname_matches)
        names_filtered = filter_matching(chain(_get_bot_hosts(), _get_walle_hosts(), _get_blocked_hostnames()))
        indices = set(_parse_ids(template, names_filtered))

        start = stage.get_data("next_id", 0)
        attempt = 0
        for next_id in (i for i in count(start) if i not in indices):
            # Periodically save current id for:
            # 1. Just in case
            # 2. In case of many computation greenlets that break ZooKeeper lock (to not start from the beginning every time
            #    we recover the lock and at least move forward after each recovery).
            # I don't expect we ever hit this code in production.
            attempt += 1
            if attempt > 100:
                log.warn("%s name allocation: tried 100 names already, still can't find a free name.", host.human_id())
                stage.set_data("next_id", next_id)
                commit_stage_changes(host, check_after=NEXT_CHECK_ADVANCE_TIME)
                return

            fqdn = template.fill(next_id)
            if not _try_assign_hostname(host, fqdn):
                next_id += 1  # try from next position
                continue

            # We must check the name individually because bot.get_hardware_info() may not return some registered names
            host_info = bot.get_host_info(fqdn)
            if host_info is None or host_info["inv"] == host.inv:
                stage.set_data("next_id", next_id + 1)  # start with next id if we have to retry later.
                log.info("'%s' hostname has been assigned to #%s.", host.name, host.inv)
                complete_current_stage(host)
                return


def _check_if_host_name_already_ok(host, template):
    if host.name is not None and template.fqdn_matches(host.name):
        log.info("%s: host already has a good name, use it.", host.human_id())
        complete_current_stage(host)
        return True

    host_info = bot.get_host_info(host.inv)
    if host_info is None:
        fail_current_stage(host, "The host is not registered in Bot.")
        return False

    # We may fail to rename host in Bot for some reason (race for example). Use next_id to generate another name that
    # will possibly succeed.

    # If host already has name try to use it instead of generation of a new one.
    if template.fqdn_matches(host_info.get("name", "")) and _try_assign_hostname(host, host_info["name"]):
        log.info("%s hostname has already been assigned to #%s.", host.name, host.inv)
        complete_current_stage(host)
        return True


@cached_with_exceptions(value_ttl=constants.MINUTE_SECONDS, error_ttl=5)
def _get_bot_hosts():
    return [host.name.lower() for host in gevent_idle_iter(bot.iter_hardware_info().iterator) if host.name is not None]


@cached_with_exceptions(value_ttl=constants.MINUTE_SECONDS, error_ttl=1)
def _get_walle_hosts():
    hostname_field = Host.name.db_field
    queryset = Host.objects.read_preference(SECONDARY_LOCAL_DC_PREFERRED)
    return [name.lower() for name in gevent_idle_iter(queryset(name__exists=True).distinct(hostname_field))]


@cached_with_exceptions(value_ttl=constants.MINUTE_SECONDS, error_ttl=1)
def _get_blocked_hostnames():
    fqdn_field = BlockedHostName.fqdn.db_field
    return [name.lower() for name in gevent_idle_iter(BlockedHostName.objects(fqdn__exists=True).distinct(fqdn_field))]


def _parse_ids(template, fqdn_list):
    for fqdn in gevent_idle_iter(fqdn_list):
        try:
            yield template.get_index(fqdn)
        except (ValueError, IndexError):
            log.error(
                "Hostname allocation internal error: got fqdn %s that doesn't match our hostname template. "
                "It should get filtered out.",
                fqdn,
            )
            raise LogicalError()


def _try_assign_hostname(host, name):
    if host.name == name:
        return True

    log.debug("Trying to assign '%s' hostname to #%s...", name, host.inv)

    cur_time = timestamp()

    if host.name is not None:
        BlockedHostName.store(host.name)

    try:
        updated = bool(Host.objects(**walle.tasks.host_query(host)).update(set__name=name, set__rename_time=cur_time))
    except mongoengine.NotUniqueError:
        return False

    if not updated:
        raise Error("Unable to commit host state change: it doesn't have the expected state already.")

    if BlockedHostName.exists(name):
        log.info("%s: failed to assign host name '%s': it became blocked during transaction.", host.human_id(), name)
        return False

    host.name = name
    host.rename_time = cur_time

    return True


# Notice: Stores context in stage data and allocates a new host on each stage retry.
register_stage(Stages.ALLOCATE_HOSTNAME, _allocate_hostname)
