"""Contains all logic for assigning OEBS project id to host in BOT."""

import logging

from sepelib.core import constants
from sepelib.core.exceptions import LogicalError
from walle.clients import bot
from walle.errors import InvalidHostConfiguration
from walle.fsm_stages.common import (
    register_stage,
    generate_stage_handler,
    get_current_stage,
    commit_stage_changes,
    complete_current_stage,
    fail_current_stage,
    retry_current_stage,
)
from walle.stages import Stages

log = logging.getLogger(__name__)

_STATUS_SWITCHING = "switching"
_STATUS_WAITING = "waiting"

_MAX_RACE_ERRORS = 10
_BOT_POLLING_PERIOD = 5

_SWITCH_TIMEOUT = 5 * constants.MINUTE_SECONDS + 2 * 1000
"""
Project assigning in BOT is done via the `rename` operation, but without actually renaming the host.

Operation running time in BOT depends on number of total rename operations hosts: they put requests to a queue and
limit number of requests to OEBS by 1 rps. Here (https://st.yandex-team.ru/BOT-1379#1461586087000) kandid@ suggests
that timeout should be 300 + 2 * $hosts.
"""

_SWITCH_RETRY_TIMEOUT = 2 * constants.HOUR_SECONDS
"""Retry every bigger interval to ensure that BOT has not lost our request."""


def _assign(host):
    stage = get_current_stage(host)
    bot_project_id = stage.get_param("bot_project_id", None)

    if not bot_project_id:
        raise LogicalError

    log.info("%s: Changing BOT/OEBS project to %s in BOT...", host.human_id(), bot_project_id)

    try:
        bot.assign_project_id(host.inv, bot_project_id)
    except InvalidHostConfiguration as e:
        log.exception("%s: Failed to set BOT/OEBS project #%s in BOT", host.human_name(), bot_project_id)
        fail_current_stage(host, "Failed to set BOT/OEBS project #{} in BOT: {}".format(bot_project_id, e))
    except bot.BotInternalError as e:
        return commit_stage_changes(host, error=str(e), check_after=_BOT_POLLING_PERIOD)
    else:
        commit_stage_changes(host, status=_STATUS_WAITING, check_now=True)


def _wait(host):
    stage = get_current_stage(host)
    bot_project_id = stage.get_param("bot_project_id", None)

    host_info = bot.get_host_info(host.inv)
    if host_info is None:
        return fail_current_stage(
            host, "The host has suddenly vanished from BOT during BOT/OEBS project change process."
        )

    if host_info.get("bot_project_id") == bot_project_id:
        return complete_current_stage(host)

    error_message = (
        "Assigning BOT/OEBS project to host takes too long. "
        "It's probably a failure on the BOT's side. "
        "Please, contact with bot@yandex-team.ru."
    )
    if get_current_stage(host).timed_out(_SWITCH_RETRY_TIMEOUT):
        return retry_current_stage(host, persistent_error=error_message + " Retrying.")

    if get_current_stage(host).timed_out(_SWITCH_TIMEOUT):
        return commit_stage_changes(host, error=error_message + " Waiting more.", check_after=_BOT_POLLING_PERIOD)

    commit_stage_changes(
        host,
        status_message="Waiting for BOT to assign the new BOT/OEBS project to the host.",
        check_after=_BOT_POLLING_PERIOD,
    )


# Attention: Retries parent stage on errors assuming that previous tasks allocate another hostname on retry.
register_stage(
    Stages.ASSIGN_BOT_PROJECT,
    generate_stage_handler(
        {
            _STATUS_SWITCHING: _assign,
            _STATUS_WAITING: _wait,
        }
    ),
    initial_status=_STATUS_SWITCHING,
)
