"""Contains all logic for assigning hostname in BOT."""

import logging

from sepelib.core import constants
from walle.clients import bot
from walle.errors import InvalidHostConfiguration, ResourceAlreadyExistsError
from walle.fsm_stages.common import (
    register_stage,
    generate_stage_handler,
    get_current_stage,
    commit_stage_changes,
    complete_current_stage,
    retry_parent_stage,
    fail_current_stage,
    increase_error_count,
    retry_current_stage,
)
from walle.models import timestamp
from walle.stages import Stages

log = logging.getLogger(__name__)

_STATUS_ASSIGN = "assign"
_STATUS_WAIT = "wait"

_MAX_RACE_ERRORS = 10
_BOT_POLLING_PERIOD = 5 * constants.MINUTE_SECONDS

_RENAME_TIMEOUT = 5 * constants.MINUTE_SECONDS + 2 * 1000
"""
Host renaming time in BOT depends on number of renaming hosts: they put requests to a queue and limit number of requests
to OEBS by 1 rps. Here (https://st.yandex-team.ru/BOT-1379#1461586087000) kandid@ suggests that timeout should be
300 + 2 * $hosts.
"""

_RENAME_RETRY_TIMEOUT = 2 * constants.HOUR_SECONDS
"""Ensure that our message did not lost every two hours."""


def _assign(host):
    log.info("%s: Assigning the hostname in BOT...", host.human_id())

    try:
        bot.rename_host(host.inv, host.name)
    except InvalidHostConfiguration as e:
        fail_current_stage(host, "Failed to assign '{}' name to the host in BOT: {}".format(host.name, e))
    except ResourceAlreadyExistsError:
        error = (
            "Failed to assign '{}' name to the host in BOT. "
            "Too many race conditions with other services: rename operation failed {} times.".format(
                host.name, _MAX_RACE_ERRORS
            )
        )

        if increase_error_count(host, get_current_stage(host), "race_errors", _MAX_RACE_ERRORS, error):
            retry_parent_stage(host)
    else:
        commit_stage_changes(host, status=_STATUS_WAIT, check_now=True)


def _wait(host):
    host_info = bot.get_host_info(host.inv)
    if host_info is None:
        return fail_current_stage(host, "The host has suddenly vanished from BOT during name assigning process.")

    if host_info.get("name") == host.name:
        host.rename_time = timestamp()
        commit_stage_changes(host, extra_fields=["rename_time"])
        return complete_current_stage(host)

    timeout_error_message = (
        "Rename operation takes too long to complete, "
        "it is probably failing ont BOT's side. "
        "Please, contact with bot@yandex-team.ru."
    )

    stage = get_current_stage(host)
    if stage.timed_out(_RENAME_RETRY_TIMEOUT):
        return retry_current_stage(host, persistent_error=timeout_error_message + " Retrying.")

    if stage.timed_out(_RENAME_TIMEOUT):
        return commit_stage_changes(
            host, error=timeout_error_message + " Waiting more.", check_after=_BOT_POLLING_PERIOD
        )

    commit_stage_changes(
        host, status_message="Waiting for BOT to assign the host name.", check_after=_BOT_POLLING_PERIOD
    )


# Attention: Retries parent stage on errors assuming that previous tasks allocate another hostname on retry.
register_stage(
    Stages.ASSIGN_HOSTNAME,
    generate_stage_handler(
        {
            _STATUS_ASSIGN: _assign,
            _STATUS_WAIT: _wait,
        }
    ),
    initial_status=_STATUS_ASSIGN,
)
